/*
 * linux/drivers/scsi/iteraid.h
 *
 * (C) Copyright 2002 ITE, inc.
 *
 * Nov 11, 2002-2003	Mark Lu	file created.
 *
 * ITE IT8212 RAID Controller device driver header for Linux
 */

#ifndef _ITERAID_H_
#define _ITERAID_H_

#include <linux/version.h>
#include <linux/types.h>
#include <linux/kdev_t.h>

#define ITE_VENDOR_ID	0x1283			/* Vendor ID (ITE)	*/
#define ITE_DEVICE_ID	0x8212			/* Device IF (IT8212)	*/
#define MAX_ADAPTERS	2			/* Max Board supported	*/
#define MAX_DEVICES	(MAX_ADAPTERS * 4)	/* Max Dev supported	*/

#define TRUE		1
#define FALSE 		0

/*
 * Undef macros which may conflict
 */
#undef	START_STOP

/************************************************************************
 * Some data types define
 ************************************************************************/
typedef long		LONG,  *PLONG;
typedef unsigned long	ULONG, *PULONG;
typedef unsigned long	DWORD, *PDWORD, BOOL;
typedef unsigned int	UINT,  *PUINT;
typedef unsigned char	BYTE,  *PBYTE;
typedef unsigned char	UCHAR, *PUCHAR;
typedef unsigned char	BOOLEAN;
typedef unsigned short	USHORT,*PUSHORT;
typedef unsigned short	WORD,  *PWORD;
typedef void		VOID,  *PVOID;
typedef unsigned char  *ADDRESS;

/************************************************************************
 * Debugging macro
 ************************************************************************/
#ifdef ITE_DEBUG
#define dprintk(msg...) printk(msg)
#else
#define dprintk(msg...) do { } while(0)
#endif

/************************************************************************
 * Raid level definitions
 ************************************************************************/
#define RAID_LEVEL_0			0
#define RAID_LEVEL_1			1
#define RAID_LEVEL_10			2
#define RAID_LEVEL_JBOD			3
#define RAID_LEVEL_NORMAL		4
#define RAID_LEVEL_NODISK		5

/************************************************************************
 * Physical disk status definitions
 ************************************************************************/
#define DISK_KEY_OFF			0
#define DISK_OFF_LINE			1
#define DISK_ON_LINE			2
#define DISK_REBUILDING			3
#define DISK_PLUGGING			4
#define DISK_PLUGGING_OK		5

#define MaximumLBAOf28Bit		0x10000000

#define DisableChannel			1
#define EnableChannel			2

#define CABLE_40_PIN			0
#define CABLE_80_PIN			1

#define RaidActive			0
#define RaidInactive			1

#define IDE_CLOCK_66			0
#define IDE_CLOCK_50			1

#define USE_ULTRA_DMA			0
#define USE_MULTIWORD_DMA		1

/************************************************************************
 *
 ************************************************************************/
typedef struct _PHYSICAL_DISK_STATUS
{
 UCHAR		ModelNumber[40];		/* Byte 00-39		*/
 ULONG		UserAddressableSectors_LOW;	/* Byte 40-43		*/
 ULONG		UserAddressableSectors_HIGH;	/* Byte 44-47		*/
 UCHAR		MultiWordDMASupport;		/* Byte 48		*/
 UCHAR		MultiWordDMAActive;		/* Byte 49		*/
 UCHAR		UltraDMASupport;		/* Byte 50		*/
 UCHAR		UltraDMAActive;			/* Byte 51		*/
 UCHAR		RaidType;			/* Byte 52		*/
 UCHAR		RaidNumber;			/* Byte 53		*/
 UCHAR		SerialNumber[20];		/* Byte 54-73		*/
 UCHAR		DiskStatus;			/* Byte 74		*/
 UCHAR		DiskOriginalLocation;		/* Byte 75		*/
 UCHAR		Cable80Pin;			/* Byte 76		*/
 UCHAR		BootableDisk;			/* Byte 77		*/
 UCHAR		StorageSize[8];			/* Byte 78-85		*/
 UCHAR		Reserved[35];			/* Byte 86-120		*/
 UCHAR		UpdateYear;			/* Byte 121		*/
 UCHAR		UpdateMonth;			/* Byte 122		*/
 UCHAR		UpdateDay;			/* Byte 123		*/
 UCHAR		FirmwareVer;			/* Byte 124		*/
 UCHAR		RebuildStatus;			/* Byte 125		*/
 UCHAR		StripeSize;			/* Byte 126		*/
 UCHAR		AutoRebuildEnable;		/* Byte 127		*/
} PHYSICAL_DISK_STATUS, *PPHYSICAL_DISK_STATUS;

/************************************************************************
 *
 ************************************************************************/
typedef struct _IT8212_SET_CHIP_STATUS_INFO
{
 USHORT		RaidType;			/* Word 129		*/
 USHORT		ContainingDisks;		/* Word 130		*/
 USHORT		UltraDmaTiming01;		/* Word 131		*/
 USHORT		UltraDmaTiming23;		/* Word 132		*/
 USHORT		UltraDmaTiming45;		/* Word 133		*/
 USHORT		UltraDmaTiming6;		/* Word 134		*/
 USHORT		MultiWordDmaTiming01;		/* Word 135		*/
 USHORT		UltraDmaTiming2;		/* Word 136		*/
 USHORT		PioTiming4;			/* Word 137		*/
 USHORT		AutoRebuildEnable;		/* Word 138		*/
 USHORT		IdeClkUDma01;			/* Word 139		*/
 USHORT		IdeClkUDma23;			/* Word 140		*/
 USHORT		IdeClkUDma45;			/* Word 141		*/
 USHORT		IdeClkUDma6;			/* Word 142		*/
 USHORT		IdeClkMDma01;			/* Word 143		*/
 USHORT		IdeClkMDma2;			/* Word 144		*/
 USHORT		IdeClkPio4;			/* Word 145		*/
 USHORT		StripeSize;			/* Word 146		*/
 USHORT		BootableDisk;			/* Word 147		*/
 USHORT		CheckHotSwapInterval;		/* Word 148		*/
 USHORT		TargetSourceDisk;		/* Word 149		*/
 USHORT		RebuildBlockSize;		/* Word 150		*/
 USHORT		ResetInterval1;			/* Word 151		*/
 USHORT		ResetInterval2;			/* Word 152		*/
 USHORT		RebuildRetryTimes;		/* Word 153		*/
 USHORT		NewlyCreated;			/* Word 154		*/
} IT8212_SET_CHIP_STATUS_INFO, *PIT8212_SET_CHIP_STATUS_INFO;

/************************************************************************
 * Serial number written to HDD (20 bytes)
 ************************************************************************/
typedef struct _RAID_SERIAL_NUMBER
{
 USHORT		Year;
 UCHAR		Month;
 UCHAR		Date;
 UCHAR		Day;
 UCHAR		Hour;
 UCHAR		Minute;
 UCHAR		Second;
 UCHAR		MiniSec;
 UCHAR		RaidType;
 UCHAR		ContainingDisks;
 UCHAR		DontCare[9];
} RAID_SERIAL_NUMBER, *PRAID_SERIAL_NUMBER;

/************************************************************************
 * Disk array create information
 *
 * Following items index definition
 * 0: Primary Master
 * 1: Secondary Master
 * 2: Primary Slave
 * 3: Secondary Slave
 ************************************************************************/
typedef struct _RAID_CREATE_INFO
{
 UCHAR			DiskArrayId;
 RAID_SERIAL_NUMBER	SerialNumber;
 UCHAR			ModelNumber[40];
 USHORT			RaidType;
 USHORT			ContainingDisks;
 USHORT			AutoRebuildEnable;
 USHORT			StripeSize;
 USHORT			BootableDisk;
 USHORT			TargetSourceDisk;
 UCHAR			ErasePartition;
 UCHAR			DMASupported[4];
 UCHAR			UDMASupported[4];
 ULONG			AddressableSectors[4];
 UCHAR			NewlyCreated;
 UCHAR			Reserved;
} RAID_CREATE_INFO, *PRAID_CREATE_INFO;

/************************************************************************
 * Rebuild data structure
 ************************************************************************/
typedef struct _RAID_REBUILD_INFO
{
 UCHAR		DiskArrayId;	/* Virtual device number (0-3)		*/
 UCHAR		SrcDisk;	/* Source disk (0-3)			*/
 UCHAR		DestDisk;	/* Destination disk (0-3)		*/
 UCHAR		Resume;		/* 1: Resume the last time rebuild	*/
				/* 0: Rebuild from LBA 0		*/
 UCHAR		Status;		/* Indicate the status of the current	*/
				/* rebuild command filled by drivers	*/
 UCHAR		Reserved[3];	/* For aligement			*/
} RAID_REBUILD_INFO, *PRAID_REBUILD_INFO;

/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

#define ITE_DRV_SIGNATURE		"ITE RAID CONTROLLER"
#define ITE_DRV_BYPASS			"ITE BYPASS MODE"

/************************************************************************
 * Extra IDE commands supported by Accusys
 ************************************************************************/
#define IDE_COMMAND_GET_CHIP_STATUS	0xFA
#define IDE_COMMAND_SET_CHIP_STATUS	0xFB
#define IDE_COMMAND_REBUILD		0xFC
#define IDE_COMMAND_REBUILD_STATUS	0xFD

#define REBUILD_ERR_WRONG_ARRAY_TYPE		0x01
#define REBUILD_ERR_DISK_TOO_SMALL		0x02
#define REBUILD_ERR_SRC_DISK_LOCATION_INCORRECT	0x03
#define REBUILD_ERR_SRC_DISK_OFFLINE		0x04
#define REBUILD_ERR_DEST_DISK_OFFLINE		0x05
#define REBUILD_ERR_DISK_BUSY			0x10	// defined by Andy, not accusys

/************************************************************************
 * ATA transfer modes
 ************************************************************************/
#define PIO_DEFAULT			0x00
#define PIO_DEFAULT_IORDY_DISABLE	0x01
#define PIO_FLOW_CONTROL		0x08
#define SINGLEWORD_DMA			0x10
#define MULTIWORD_DMA			0x20
#define ULTRA_DMA			0x40

/************************************************************************
 * IDE registers offset
 ************************************************************************/
#define IDE_NR_PORTS			10

#define IDE_DATA_OFFSET			0
#define IDE_ERROR_OFFSET		1
#define IDE_NSECTOR_OFFSET		2
#define IDE_LOCYL_OFFSET		3
#define IDE_MIDCYL_OFFSET		4
#define IDE_HCYL_OFFSET			5
#define IDE_SELECT_OFFSET		6
#define IDE_STATUS_OFFSET		7
#define IDE_CONTROL_OFFSET		8
#define IDE_IRQ_OFFSET			9

#define	IDE_FEATURE_OFFSET		IDE_ERROR_OFFSET
#define IDE_COMMAND_OFFSET		IDE_STATUS_OFFSET
#define IDE_ALTERNATE_OFFSET		IDE_CONTROL_OFFSET

/************************************************************************
 * ATAPI registers offset
 ************************************************************************/
#define ATAPI_DATA_OFFSET		0
#define ATAPI_ERROR_OFFSET		1
#define ATAPI_INTREASON_OFFSET		2
#define ATAPI_UNUSED1_OFFSET		3
#define ATAPI_LCYL_OFFSET		4
#define ATAPI_HCYL_OFFSET		5
#define ATAPI_SELECT_OFFSET		6
#define ATAPI_STATUS_OFFSET		7
#define ATAPI_CONTROL_OFFSET		8

#define ATAPI_COMMAND_OFFSET		ATAPI_STATUS_OFFSET
#define	ATAPI_FEATURE_OFFSET		ATAPI_ERROR_OFFSET

/************************************************************************
 * Following structures are according to SPC-3 (by Chanel)
 ************************************************************************/
typedef struct _SCSI_MODE_SENSE6
{
 UCHAR		OperationCode;
 UCHAR		Reserved1 : 3;
 UCHAR		Dbd : 1;
 UCHAR		Reserved2 : 4;
 UCHAR		PageCode : 6;
 UCHAR		Pc : 2;
 UCHAR		SubpageCode;
 UCHAR		AllocationLength;
 UCHAR		Control;
} SCSI_MODE_SENSE6, *PSCSI_MODE_SENSE6;

typedef struct _SCSI_MODE_SENSE10
{
 UCHAR		OperationCode;
 UCHAR		Reserved1 : 3;
 UCHAR		Dbd : 1;
 UCHAR		LLBAA : 1;
 UCHAR		Reserved2 : 3;
 UCHAR		PageCode : 6;
 UCHAR		Pc : 2;
 UCHAR		SubpageCode;
 UCHAR		Reserved3[3];
 UCHAR		AllocationLengthMsb;
 UCHAR		AllocationLengthLsb;
 UCHAR		Control;
} SCSI_MODE_SENSE10, *PSCSI_MODE_SENSE10;

typedef struct _SCSI_MODE_SELECT6
{
 UCHAR		OperationCode;
 UCHAR		SPBit : 1;
 UCHAR		Reserved1 : 3;
 UCHAR		PFBit : 1;
 UCHAR		Reserved2 : 3;
 UCHAR		Reserved3[2];
 UCHAR		ParameterListLength;
 UCHAR		Control;
} SCSI_MODE_SELECT6, *PSCSI_MODE_SELECT6;

typedef struct _SCSI_MODE_SELECT10
{
 UCHAR		OperationCode;
 UCHAR		SPBit : 1;
 UCHAR		Reserved1 : 3;
 UCHAR		PFBit : 1;
 UCHAR		Reserved2 : 3;
 UCHAR		Reserved3[5];
 UCHAR		ParameterListLengthMsb;
 UCHAR		ParameterListLengthLsb;
 UCHAR		Control;
} SCSI_MODE_SELECT10, *PSCSI_MODE_SELECT10;

typedef struct _SCSI_MODE_PARAMETER_HEADER6
{
 UCHAR		ModeDataLength;
 UCHAR		MediumType;
 UCHAR		DeviceSpecificParameter;
 UCHAR		BlockDescriptorLength;
} SCSI_MODE_PARAMETER_HEADER6, *PSCSI_MODE_PARAMETER_HEADER6;

typedef struct _SCSI_MODE_PARAMETER_HEADER10
{
 UCHAR		ModeDataLengthMsb;
 UCHAR		ModeDataLengthLsb;
 UCHAR		MediumType;
 UCHAR		DeviceSpecificParameter;
 UCHAR		Reserved[2];
 UCHAR		BlockDescriptorLengthMsb;
 UCHAR		BlockDescriptorLengthLsb;
} SCSI_MODE_PARAMETER_HEADER10, *PSCSI_MODE_PARAMETER_HEADER10;

typedef struct _SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER
{
 UCHAR		DesityCode;
 UCHAR		NumberOfBlocks2;
 UCHAR		NumberOfBlocks1;
 UCHAR		NumberOfBlocks0;
 UCHAR		Reserved;
 UCHAR		BlockLength2;
 UCHAR		BlockLength1;
 UCHAR		BlockLength0;
} SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER, *PSCSI_MODE_PARAMTER_BLOCK_DESCRIPTER;

/************************************************************************
 * IDE command definitions
 ************************************************************************/
#define IDE_COMMAND_ATAPI_RESET			0x08
#define IDE_COMMAND_RECALIBRATE			0x10
#define IDE_COMMAND_READ_SECTOR			0x20
#define IDE_COMMAND_READ_SECTOR_EXT		0x24
#define IDE_COMMAND_READ_DMA_EXT		0x25
#define IDE_COMMAND_READ_MULTIPLE_EXT		0x29
#define IDE_COMMAND_WRITE_SECTOR		0x30
#define IDE_COMMAND_WRITE_SECTOR_EXT		0x34
#define IDE_COMMAND_WRITE_DMA_EXT		0x35
#define IDE_COMMAND_WRITE_MULTIPLE_EXT		0x39
#define IDE_COMMAND_READ_VERIFY			0x40
#define IDE_COMMAND_READ_VERIFY_EXT		0x42
#define IDE_COMMAND_SEEK			0x70
#define IDE_COMMAND_SET_DRIVE_PARAMETERS	0x91
#define IDE_COMMAND_ATAPI_PACKET		0xA0
#define IDE_COMMAND_ATAPI_IDENTIFY		0xA1
#define IDE_COMMAND_READ_MULTIPLE		0xC4
#define IDE_COMMAND_WRITE_MULTIPLE		0xC5
#define IDE_COMMAND_SET_MULTIPLE		0xC6
#define IDE_COMMAND_READ_DMA			0xC8
#define IDE_COMMAND_WRITE_DMA			0xCA
#define IDE_COMMAND_GET_MEDIA_STATUS		0xDA
#define IDE_COMMAND_ENABLE_MEDIA_STATUS		0xEF
#define IDE_COMMAND_SET_FEATURE			0xEF
#define IDE_COMMAND_IDENTIFY			0xEC
#define IDE_COMMAND_MEDIA_EJECT			0xED

/************************************************************************
 * IDE status definitions
 ************************************************************************/
#define IDE_STATUS_ERROR			0x01
#define IDE_STATUS_INDEX			0x02
#define IDE_STATUS_CORRECTED_ERROR		0x04
#define IDE_STATUS_DRQ				0x08
#define IDE_STATUS_DSC				0x10
#define IDE_STATUS_DRDY				0x40
#define IDE_STATUS_IDLE				0x50
#define IDE_STATUS_BUSY				0x80

/************************************************************************
 * IDE drive control definitions
 ************************************************************************/
#define IDE_DC_DISABLE_INTERRUPTS		0x02
#define IDE_DC_RESET_CONTROLLER			0x04
#define IDE_DC_REENABLE_CONTROLLER		0x00

/************************************************************************
 * IDE error definitions
 ************************************************************************/
#define IDE_ERROR_BAD_BLOCK			0x80
#define IDE_ERROR_DATA_ERROR			0x40
#define IDE_ERROR_MEDIA_CHANGE			0x20
#define IDE_ERROR_ID_NOT_FOUND			0x10
#define IDE_ERROR_MEDIA_CHANGE_REQ		0x08
#define IDE_ERROR_COMMAND_ABORTED		0x04
#define IDE_ERROR_END_OF_MEDIA			0x02
#define IDE_ERROR_ILLEGAL_LENGTH		0x01

/************************************************************************
 * IDENTIFY data
 ************************************************************************/
typedef struct _IDENTIFY_DATA
{
 USHORT		GeneralConfiguration;		// 00 00
 USHORT		NumberOfCylinders;		// 02  1
 USHORT		Reserved1;			// 04  2
 USHORT		NumberOfHeads;			// 06  3
 USHORT		UnformattedBytesPerTrack;	// 08  4
 USHORT		UnformattedBytesPerSector;	// 0A  5
 USHORT		SectorsPerTrack;		// 0C  6
 USHORT		VendorUnique1[3];		// 0E  7-9
 USHORT		SerialNumber[10];		// 14  10-19
 USHORT		BufferType;			// 28  20
 USHORT		BufferSectorSize;		// 2A  21
 USHORT		NumberOfEccBytes;		// 2C  22
 USHORT		FirmwareRevision[4];		// 2E  23-26
 USHORT		ModelNumber[20];		// 36  27-46
 UCHAR		MaximumBlockTransfer;		// 5E  47
 UCHAR		VendorUnique2;			// 5F
 USHORT		DoubleWordIo;			// 60  48
 USHORT		Capabilities;			// 62  49
 USHORT		Reserved2;			// 64  50
 UCHAR		VendorUnique3;			// 66  51
 UCHAR		PioCycleTimingMode;		// 67
 UCHAR		VendorUnique4;			// 68  52
 UCHAR		DmaCycleTimingMode;		// 69
 USHORT		TranslationFieldsValid : 1;	// 6A  53
 USHORT		Reserved3 : 15;			//
 USHORT		NumberOfCurrentCylinders;	// 6C  54
 USHORT		NumberOfCurrentHeads;		// 6E  55
 USHORT		CurrentSectorsPerTrack;		// 70  56
 ULONG		CurrentSectorCapacity;		// 72  57-58
 USHORT		CurrentMultiSectorSetting;	//     59
 ULONG		UserAddressableSectors;		//     60-61
 USHORT		SingleWordDMASupport : 8;	//     62
 USHORT		SingleWordDMAActive : 8;	//
 USHORT		MultiWordDMASupport : 8;	//     63
 USHORT		MultiWordDMAActive : 8;		//
 USHORT		AdvancedPIOModes : 8;		//     64
 USHORT		Reserved4 : 8;			//
 USHORT		MinimumMWXferCycleTime;		//     65
 USHORT		RecommendedMWXferCycleTime;	//     66
 USHORT		MinimumPIOCycleTime;		//     67
 USHORT		MinimumPIOCycleTimeIORDY;	//     68
 USHORT		Reserved5[2];			//     69-70
 USHORT		ReleaseTimeOverlapped;		//     71
 USHORT		ReleaseTimeServiceCommand;	//     72
 USHORT		MajorRevision;			//     73
 USHORT		MinorRevision;			//     74
 USHORT		Reserved6[50];			//     75-126
 USHORT		SpecialFunctionsEnabled;	//     127
 USHORT		Reserved7[128];			//     128-255
} IDENTIFY_DATA, *PIDENTIFY_DATA;

/************************************************************************
 * Identify data without the Reserved4
 ************************************************************************/
typedef struct _IDENTIFY_DATA2
{
 USHORT		GeneralConfiguration;		// 00
 USHORT		NumberOfCylinders;		// 01
 USHORT		Reserved1;			// 02
 USHORT		NumberOfHeads;			// 03
 USHORT		Reserved2[2];			// 04-05
 USHORT		SectorsPerTrack;		// 06
 USHORT		Reserved3[3];			// 07-09
 USHORT		SerialNumber[10];		// 10-19
 USHORT		Reserved4[3];			// 20-22
 USHORT		FirmwareRevision[4];		// 23-26
 USHORT		ModelNumber[20];		// 27-46
 USHORT		MaximumBlockTransfer;		// 47
 USHORT		Reserved5;			// 48
 USHORT		Capabilities[2];		// 49-50
 USHORT		Reserved6[2];			// 51-52
 USHORT		ValidFieldIndicator;		// 53
 USHORT		NumberOfCurrentCylinders;	// 54
 USHORT		NumberOfCurrentHeads;		// 55
 USHORT		CurrentSectorsPerTrack;		// 56
 USHORT		CurrentSectorCapacityLow;	// 57
 USHORT		CurrentSectorCapacityHigh;	// 58
 USHORT		CurrentMultiSectorSetting;	// 59
 ULONG		UserAddressableSectors;		// 60-61
 USHORT		Reserved7;			// 62
 UCHAR		MultiWordDMASupport;		// 63
 UCHAR		MultiWordDMAActive;		//
 USHORT		AdvancedPIOModes;		// 64
 USHORT		MinimumMWXferCycleTime;		// 65
 USHORT		RecommendedMWXferCycleTime;	// 66
 USHORT		MinimumPIOCycleTime;		// 67
 USHORT		MinimumPIOCycleTimeIORDY;	// 68
 USHORT		Reserved8[6];			// 69-74
 USHORT		QueueDepth;			// 75
 USHORT		Reserved9[4];			// 76-79
 USHORT		MajorVersionNumber;		// 80
 USHORT		MinorVersionNumber;		// 81
 ULONG		CmdSetSupported;		// 82-83
 USHORT		CmdSetFeatureSupportedExt;	// 84
 USHORT		CmdSetFeatureEnabledLow;	// 85
 USHORT		CmdSetFeatureEnabledHigh;	// 86
 USHORT		CmdSetFeatureDefault;		// 87
 UCHAR		UltraDMASupport;		// 88
 UCHAR		UltraDMAActive;			//			
 USHORT		SecurityEraseTime;		// 89
 USHORT		EnhancedSecurityEraseTime;	// 90
 USHORT		PowerManagementValue;		// 91
 USHORT		MasterPasswordRevision;		// 92
 USHORT		HwResetResult;			// 93
 USHORT		Reserved11[6];			// 94-99
 ULONG		Capacity_48bit_LOW;		// 100-101
 ULONG		Capacity_48bit_HIGH;		// 102-103
 USHORT		Reserved12[24];			// 104-127
 USHORT		SecurityStatus;			// 128
 USHORT		Reserved13[31];			// 129-159 vendor specific
 USHORT		Reserved14[96];			// 160-255
} IDENTIFY_DATA2, *PIDENTIFY_DATA2;

#define IDENTIFY_DATA_SIZE sizeof(IDENTIFY_DATA)

/************************************************************************
 * IDENTIFY capability bit definitions.
 ************************************************************************/
#define IDENTIFY_CAPABILITIES_DMA_SUPPORTED	0x0100
#define IDENTIFY_CAPABILITIES_LBA_SUPPORTED	0x0200

/************************************************************************
 * IDENTIFY DMA timing cycle modes.
 ************************************************************************/
#define IDENTIFY_DMA_CYCLES_MODE_0		0x00
#define IDENTIFY_DMA_CYCLES_MODE_1		0x01
#define IDENTIFY_DMA_CYCLES_MODE_2		0x02

typedef struct _SENSE_DATA
{
 UCHAR		ErrorCode:7;
 UCHAR		Valid:1;
 UCHAR		SegmentNumber;
 UCHAR		SenseKey:4;
 UCHAR		Reserved:1;
 UCHAR		IncorrectLength:1;
 UCHAR		EndOfMedia:1;
 UCHAR		FileMark:1;
 UCHAR		Information[4];
 UCHAR		AdditionalSenseLength;
 UCHAR		CommandSpecificInformation[4];
 UCHAR		AdditionalSenseCode;
 UCHAR		AdditionalSenseCodeQualifier;
 UCHAR		FieldReplaceableUnitCode;
 UCHAR		SenseKeySpecific[3];
} SENSE_DATA, *PSENSE_DATA;

/************************************************************************
 * Sense codes
 ************************************************************************/
#define SCSI_SENSE_NO_SENSE		0x00
#define SCSI_SENSE_RECOVERED_ERROR	0x01
#define SCSI_SENSE_NOT_READY		0x02
#define SCSI_SENSE_MEDIUM_ERROR		0x03
#define SCSI_SENSE_HARDWARE_ERROR	0x04
#define SCSI_SENSE_ILLEGAL_REQUEST	0x05
#define SCSI_SENSE_UNIT_ATTENTION	0x06
#define SCSI_SENSE_DATA_PROTECT		0x07
#define SCSI_SENSE_BLANK_CHECK		0x08
#define SCSI_SENSE_UNIQUE		0x09
#define SCSI_SENSE_COPY_ABORTED		0x0A
#define SCSI_SENSE_ABORTED_COMMAND	0x0B
#define SCSI_SENSE_EQUAL		0x0C
#define SCSI_SENSE_VOL_OVERFLOW		0x0D
#define SCSI_SENSE_MISCOMPARE		0x0E
#define SCSI_SENSE_RESERVED		0x0F

/************************************************************************
 * Additional Sense codes
 ************************************************************************/
#define SCSI_ADSENSE_NO_SENSE		0x00
#define SCSI_ADSENSE_MAN_INTERV		0x03
#define SCSI_ADSENSE_LUN_NOT_READY	0x04
#define SCSI_ADSENSE_ILLEGAL_COMMAND	0x20
#define SCSI_ADSENSE_ILLEGAL_BLOCK	0x21
#define SCSI_ADSENSE_INVALID_LUN	0x25
#define SCSI_ADSENSE_SELECT_TIMEOUT	0x45
#define SCSI_ADSENSE_MUSIC_AREA		0xA0
#define SCSI_ADSENSE_DATA_AREA		0xA1
#define SCSI_ADSENSE_VOLUME_OVERFLOW	0xA7

#define SCSI_ADSENSE_NO_MEDIA_IN_DEVICE	0x3A
#define SCSI_ADWRITE_PROTECT		0x27
#define SCSI_ADSENSE_MEDIUM_CHANGED	0x28
#define SCSI_ADSENSE_BUS_RESET		0x29
#define SCSI_ADSENSE_TRACK_ERROR	0x14
#define SCSI_ADSENSE_SEEK_ERROR		0x15
#define SCSI_ADSENSE_REC_DATA_NOECC	0x17
#define SCSI_ADSENSE_REC_DATA_ECC	0x18
#define SCSI_ADSENSE_ILLEGAL_MODE	0x64
#define SCSI_ADSENSE_BAD_CDB		0x24
#define SCSI_ADSENSE_BAD_PARM_LIST	0x26
#define SCSI_ADSENSE_CANNOT_READ_MEDIUM	0x30

#define SCSISTAT_CHECK_CONDITION	0x02

/************************************************************************
 * Inquiry buffer structure. This is the data returned from the target
 * after it receives an inquiry.
 *
 * This structure may be extended by the number of bytes specified
 * in the field AdditionalLength. The defined size constant only
 * includes fields through ProductRevisionLevel.
 *
 * The NT SCSI drivers are only interested in the first 36 bytes of data.
 ************************************************************************/

#define INQUIRYDATABUFFERSIZE 36

typedef struct _INQUIRYDATA
{
 UCHAR		DeviceType : 5;
 UCHAR		DeviceTypeQualifier : 3;
 UCHAR		DeviceTypeModifier : 7;
 UCHAR		RemovableMedia : 1;
 UCHAR		Versions;
 UCHAR		ResponseDataFormat;
 UCHAR		AdditionalLength;
 UCHAR		Reserved[2];
 UCHAR		SoftReset : 1;
 UCHAR		CommandQueue : 1;
 UCHAR		Reserved2 : 1;
 UCHAR		LinkedCommands : 1;
 UCHAR		Synchronous : 1;
 UCHAR		Wide16Bit : 1;
 UCHAR		Wide32Bit : 1;
 UCHAR		RelativeAddressing : 1;
 UCHAR		VendorId[8];
 UCHAR		ProductId[16];
 UCHAR		ProductRevisionLevel[4];
 UCHAR		VendorSpecific[20];
 UCHAR		Reserved3[40];
} INQUIRYDATA, *PINQUIRYDATA;

#define DIRECT_ACCESS_DEVICE		0x00	/* Disks		*/

/************************************************************************
 * Read Capacity Data - returned in Big Endian format
 ************************************************************************/
typedef struct _READ_CAPACITY_DATA
{
 ULONG		LogicalBlockAddress;
 ULONG		BytesPerBlock;
} READ_CAPACITY_DATA, *PREAD_CAPACITY_DATA;

#define MAXIMUM_CDB_SIZE	12

/************************************************************************
 * CDB (Command Descriptor Block)
 ************************************************************************/
typedef union _CDB
{
 /*
  * Standard 6-byte CDB
  */
 struct _CDB6READWRITE
    {
     UCHAR	OperationCode;
     UCHAR	LogicalBlockMsb1 : 5;
     UCHAR	LogicalUnitNumber : 3;
     UCHAR	LogicalBlockMsb0;
     UCHAR	LogicalBlockLsb;
     UCHAR	TransferBlocks;
     UCHAR	Control;
    } CDB6READWRITE, *PCDB6READWRITE;

 /*
  * Standard 10-byte CDB
  */
 struct _CDB10
    {
     UCHAR	OperationCode;
     UCHAR	Reserved1 : 5;
     UCHAR	LogicalUnitNumber : 3;
     UCHAR	LogicalBlockByte0;
     UCHAR	LogicalBlockByte1;
     UCHAR	LogicalBlockByte2;
     UCHAR	LogicalBlockByte3;
     UCHAR	Reserved2;
     UCHAR	TransferBlocksMsb;
     UCHAR	TransferBlocksLsb;
     UCHAR	Control;
    } CDB10, *PCDB10;

 struct _START_STOP
    {
     UCHAR	OperationCode;
     UCHAR	Immediate: 1;
     UCHAR	Reserved1 : 4;
     UCHAR	LogicalUnitNumber : 3;
     UCHAR	Reserved2[2];
     UCHAR	Start : 1;
     UCHAR	LoadEject : 1;
     UCHAR	Reserved3 : 6;
     UCHAR	Control;
    } START_STOP, *PSTART_STOP;

} CDB, *PCDB;

/************************************************************************
 * SCSI CDB operation codes
 ************************************************************************/
#define SCSIOP_TEST_UNIT_READY		0x00
#define SCSIOP_REZERO_UNIT		0x01
#define SCSIOP_REWIND			0x01
#define SCSIOP_REQUEST_BLOCK_ADDR	0x02
#define SCSIOP_REQUEST_SENSE		0x03
#define SCSIOP_FORMAT_UNIT		0x04
#define SCSIOP_READ_BLOCK_LIMITS	0x05
#define SCSIOP_REASSIGN_BLOCKS		0x07
#define SCSIOP_READ6			0x08
#define SCSIOP_RECEIVE			0x08
#define SCSIOP_WRITE6			0x0A
#define SCSIOP_PRINT			0x0A
#define SCSIOP_SEND			0x0A
#define SCSIOP_SEEK6			0x0B
#define SCSIOP_TRACK_SELECT		0x0B
#define SCSIOP_SLEW_PRINT		0x0B
#define SCSIOP_SEEK_BLOCK		0x0C
#define SCSIOP_PARTITION		0x0D
#define SCSIOP_READ_REVERSE		0x0F
#define SCSIOP_WRITE_FILEMARKS		0x10
#define SCSIOP_FLUSH_BUFFER		0x10
#define SCSIOP_SPACE			0x11
#define SCSIOP_INQUIRY			0x12
#define SCSIOP_VERIFY6			0x13
#define SCSIOP_RECOVER_BUF_DATA		0x14
#define SCSIOP_MODE_SELECT		0x15
#define SCSIOP_RESERVE_UNIT		0x16
#define SCSIOP_RELEASE_UNIT		0x17
#define SCSIOP_COPY			0x18
#define SCSIOP_ERASE			0x19
#define SCSIOP_MODE_SENSE		0x1A
#define SCSIOP_START_STOP_UNIT		0x1B
#define SCSIOP_STOP_PRINT		0x1B
#define SCSIOP_LOAD_UNLOAD		0x1B
#define SCSIOP_RECEIVE_DIAGNOSTIC	0x1C
#define SCSIOP_SEND_DIAGNOSTIC		0x1D
#define SCSIOP_MEDIUM_REMOVAL		0x1E
#define SCSIOP_READ_CAPACITY		0x25
#define SCSIOP_READ			0x28
#define SCSIOP_WRITE			0x2A
#define SCSIOP_SEEK			0x2B
#define SCSIOP_LOCATE			0x2B
#define SCSIOP_WRITE_VERIFY		0x2E
#define SCSIOP_VERIFY			0x2F
#define SCSIOP_SEARCH_DATA_HIGH		0x30
#define SCSIOP_SEARCH_DATA_EQUAL	0x31
#define SCSIOP_SEARCH_DATA_LOW		0x32
#define SCSIOP_SET_LIMITS		0x33
#define SCSIOP_READ_POSITION		0x34
#define SCSIOP_SYNCHRONIZE_CACHE	0x35
#define SCSIOP_COMPARE			0x39
#define SCSIOP_COPY_COMPARE		0x3A
#define SCSIOP_WRITE_DATA_BUFF		0x3B
#define SCSIOP_READ_DATA_BUFF		0x3C
#define SCSIOP_CHANGE_DEFINITION	0x40
#define SCSIOP_READ_SUB_CHANNEL		0x42
#define SCSIOP_READ_TOC			0x43
#define SCSIOP_READ_HEADER		0x44
#define SCSIOP_PLAY_AUDIO		0x45
#define SCSIOP_PLAY_AUDIO_MSF		0x47
#define SCSIOP_PLAY_TRACK_INDEX		0x48
#define SCSIOP_PLAY_TRACK_RELATIVE	0x49
#define SCSIOP_PAUSE_RESUME		0x4B
#define SCSIOP_LOG_SELECT		0x4C
#define SCSIOP_LOG_SENSE		0x4D
#define SCSIOP_MODE_SELECT10		0x55
#define SCSIOP_MODE_SENSE10		0x5A
#define SCSIOP_LOAD_UNLOAD_SLOT		0xA6
#define SCSIOP_MECHANISM_STATUS		0xBD
#define SCSIOP_READ_CD			0xBE

#define DRIVER_NAME		"Device Driver for IT8212 RAID Controller"
#define COMPANY_NAME		"Integrated Technology Express, Inc."
#define CONTROLLER_NAME_IT8212	"IT8212 UDMA/ATA133 RAID Controller"
#define PROC_DIR_NAME		"it8212"
#define ITE_MAX_CMDS		124

#define PCI_IOSEN	0x01	/* Enable IO space			*/
#define PCI_BMEN	0x04	/* Enable IDE bus master		*/

/************************************************************************
 * PRD (Physical Region Descriptor) = Scatter-gather table
 *
 * |  byte3   |	 byte2	 |  byte1   |   byte0   |
 * +--------------------------------------------+
 * | Memory Region Physical Base Address[31:1]  |
 * +----+----------------+----------------------+
 * |EOT |  reserved      |   Byte count[15:1]   |
 * +----+----------------+----------------------+
 ************************************************************************/
typedef struct _PRD_TABLE_ENTRY
{
 ULONG		PhysicalBaseAddress;		/* Byte0 - Byte3	*/
 USHORT		ByteCount;			/* Byte4 - Byte5	*/
 USHORT		EndOfTable;			/* Byte6 - Byte7	*/
} PRD_TABLE_ENTRY, *PPRD_TABLE_ENTRY;

#define SG_FLAG_EOT			0x8000	/* End of PRD		*/
#define MAX_SG_DESCRIPTORS		17	/* 17 -- maximum 64K	*/

#define NUM_OF_PRD_TABLE_ENTRY		0x10

/************************************************************************
 * Bus master register bits definition
 ************************************************************************/
#define BM_CMD_FLG_START		0x01
#define BM_CMD_FLG_WRTTOMEM		0x08
#define BM_CMD_FLG_WRTTODSK		0x00

#define BM_STAT_FLG_ACTIVE		0x01
#define BM_STAT_FLG_ERR			0x02
#define BM_STAT_FLG_INT			0x04		
#define BM_DRV0_DMA_CAPABLE		0x20
#define BM_DRV1_DMA_CAPABLE		0x40

#define BM_PRD_FLG_EOT  		0x8000

/************************************************************************
 * SRB Functions
 ************************************************************************/
#define SRB_FUNCTION_EXECUTE_SCSI	0x00
#define SRB_FUNCTION_IO_CONTROL		0x02
#define SRB_FUNCTION_SHUTDOWN		0x07
#define SRB_FUNCTION_FLUSH		0x08

/************************************************************************
 * SRB Status
 ************************************************************************/
#define SRB_STATUS_PENDING		0x00
#define SRB_STATUS_SUCCESS		0x01
#define SRB_STATUS_ABORTED		0x02
#define SRB_STATUS_ABORT_FAILED		0x03
#define SRB_STATUS_ERROR		0x04
#define SRB_STATUS_BUSY			0x05
#define SRB_STATUS_INVALID_REQUEST	0x06
#define SRB_STATUS_INVALID_PATH_ID	0x07
#define SRB_STATUS_NO_DEVICE		0x08
#define SRB_STATUS_TIMEOUT		0x09
#define SRB_STATUS_SELECTION_TIMEOUT	0x0A
#define SRB_STATUS_COMMAND_TIMEOUT	0x0B
#define SRB_STATUS_MESSAGE_REJECTED	0x0D
#define SRB_STATUS_BUS_RESET		0x0E
#define SRB_STATUS_PARITY_ERROR		0x0F
#define SRB_STATUS_REQUEST_SENSE_FAILED	0x10
#define SRB_STATUS_NO_HBA		0x11
#define SRB_STATUS_DATA_OVERRUN		0x12
#define SRB_STATUS_UNEXPECTED_BUS_FREE	0x13
#define SRB_STATUS_BAD_SRB_BLOCK_LENGTH	0x15
#define SRB_STATUS_REQUEST_FLUSHED	0x16
#define SRB_STATUS_INVALID_LUN		0x20
#define SRB_STATUS_INVALID_TARGET_ID	0x21
#define SRB_STATUS_BAD_FUNCTION		0x22
#define SRB_STATUS_ERROR_RECOVERY	0x23
#define SRB_STATUS_NEED_REQUEUE		0x24

/************************************************************************
 * SRB Status Masks
 ************************************************************************/
#define SRB_STATUS_QUEUE_FROZEN		0x40
#define SRB_STATUS_AUTOSENSE_VALID	0x80

#define SRB_STATUS(Status)	\
	(Status & ~(SRB_STATUS_AUTOSENSE_VALID | SRB_STATUS_QUEUE_FROZEN))

/************************************************************************
 * SRB Flag Bits
 ************************************************************************/
#define SRB_FLAGS_DATA_IN		0x00000040
#define SRB_FLAGS_DATA_OUT		0x00000080

/************************************************************************
 * SRB Working flags define area
 ************************************************************************/
#define	SRB_WFLAGS_USE_INTERNAL_BUFFER	0x00000001
#define	SRB_WFLAGS_IGNORE_ARRAY		0x00000002
#define	SRB_WFLAGS_HAS_CALL_BACK	0x00000004
#define	SRB_WFLAGS_MUST_DONE		0x00000008
#define	SRB_WFLAGS_ON_MIRROR_DISK	0x00000010
#define	SRB_WFLAGS_ON_SOURCE_DISK	0x00000020
#define SRB_WFLAGS_ARRAY_IO_STARTED	0x10000000
#define SRB_WFLAGS_WATCHTIMER_CALLED	0x20000000
#define SRB_WFLAGS_USE_SG		0x40000000

/************************************************************************
 * SCSI I/O Request Block
 ************************************************************************/
typedef struct _SCSI_REQUEST_BLOCK
{
 USHORT			Length;
 UCHAR			Function;
 UCHAR			SrbStatus;
 UCHAR			ScsiStatus;
 UCHAR			TargetId;
 UCHAR			Lun;
 UCHAR			CdbLength;
 UCHAR			SenseInfoBufferLength;
 UCHAR			UseSg;
 UCHAR			reseved[2];
 ULONG			WorkingFlags;
 ULONG			SrbFlags;
 ULONG			DataTransferLength;
 ULONG			TimeOutValue;
 PVOID			DataBuffer;
 PVOID			SenseInfoBuffer;
 UCHAR			Cdb[16];
 Scsi_Cmnd *		pREQ;
} SCSI_REQUEST_BLOCK, *PSCSI_REQUEST_BLOCK;

#define SCSI_REQUEST_BLOCK_SIZE sizeof(SCSI_REQUEST_BLOCK)

/************************************************************************
 * Second device flags
 ***********************************************************************/
#define DFLAGS_REDUCE_MODE	        0x00010000
#define DFLAGS_DEVICE_DISABLED		0x00020000	
#define DFLAGS_BOOTABLE_DEVICE		0x00080000	
#define DFLAGS_BOOT_MARK		0x00100000
#define DFLAGS_NEW_ADDED		0x40000000
#define DFLAGS_REMAINED_MEMBER		0x80000000

/************************************************************************
 * Device Extension Device Flags
 ************************************************************************/
/*
 * Indicates that some device is present.
 */
#define DFLAGS_DEVICE_PRESENT		0x0001

/*
 * Indicates whether ATAPI commands can be used.
 */
#define DFLAGS_ATAPI_DEVICE		0x0002

/*
 * Indicates whether this is a tape device.
 */
#define DFLAGS_TAPE_DEVICE		0x0004

/*
 * Indicates whether device interrupts as DRQ is set after
 * receiving Atapi Packet Command.
 */
#define DFLAGS_INT_DRQ			0x0008

/*
 * Indicates that the drive has the 'removable' bit set in
 * identify data (offset 128)
 */
#define DFLAGS_REMOVABLE_DRIVE		0x0010

/*
 * Media status notification enabled.
 */
#define DFLAGS_MEDIA_STATUS_ENABLED	0x0020

/*
 * Indicates atapi 2.5 changer present.
 */
#define DFLAGS_ATAPI_CHANGER		0x0040

/*
 * Indicates multi-platter device, not conforming to the 2.5 spec.
 */
#define DFLAGS_SANYO_ATAPI_CHANGER	0x0080

/*
 * Indicates that the init path for changers has already been done.
 */
#define DFLAGS_CHANGER_INITED		0x0100
#define DFLAGS_CONFIG_CHANGED		0x0200

#define UDMA_MODE_5_6			0x80

/************************************************************************
 * Used to disable 'advanced' features.
 ************************************************************************/
#define MAX_ERRORS			4

/************************************************************************
 * ATAPI command definitions
 ************************************************************************/
#define ATAPI_MODE_SENSE		0x5A
#define ATAPI_MODE_SELECT		0x55
#define ATAPI_FORMAT_UNIT		0x24

/************************************************************************
 * User IOCTL structure
 * Notes:
 * (1) Data transfers are limited to PAGE_SIZE (4k on i386, 8k for alpha)
 ************************************************************************/
typedef struct	_uioctl_t
{
 USHORT		inlen;		/* Length of data written to device	*/
 USHORT		outlen;		/* Length of data read from device	*/
 void *		data;		/* Data read from devic starts here	*/
 UCHAR		status;		/* Status return from driver		*/
 UCHAR		reserved[3];	/* For 4-byte alignment			*/
}		uioctl_t;

/************************************************************************
 * IOCTL commands for RAID
 ************************************************************************/
#define ITE_IOCMAGIC			't'

#define ITE_IOC_GET_PHY_DISK_STATUS	_IO(ITE_IOCMAGIC, 1)
#define ITE_IOC_CREATE_DISK_ARRAY	_IO(ITE_IOCMAGIC, 2)
#define ITE_IOC_REBUILD_START		_IO(ITE_IOCMAGIC, 3)
#define ITE_IOC_GET_REBUILD_STATUS	_IO(ITE_IOCMAGIC, 4)
#define ITE_IOC_RESET_ADAPTER		_IO(ITE_IOCMAGIC, 5)
#define ITE_IOC_GET_DRIVER_VERSION	_IO(ITE_IOCMAGIC, 6)

/************************************************************************
 * _Channel
 ************************************************************************/
typedef struct _Channel
{
 /*
  * IDE (ATAPI) io port address.
  */
 ULONG			io_ports[IDE_NR_PORTS];	

 /*
  * DMA base address.
  */
 ULONG			dma_base;

 /*
  * Flags word for each possible device.
  */
 USHORT			DeviceFlags[2];

 /*
  * Indicates number of platters on changer-ish devices.
  */	
 ULONG			DiscsPresent[2];

 /*
  * Indicates expecting an interrupt.
  */
 BOOLEAN		ExpectingInterrupt;

 /*
  * Indicate last tape command was DSC Restrictive.
  */
 BOOLEAN		RDP;

 /*
  * Interrupt level.
  */
 UCHAR			InterruptLevel;

 /*
  * Placeholder for status register after a GET_MEDIA_STATUS command.
  */
 UCHAR			ReturningMediaStatus;

 /*
  * Remember the channel number (0, 1)
  */
 UCHAR			channel;

 /*
  * Indicates cable status.
  */
 UCHAR			Cable80[2];

 /*
  * Data buffer pointer.
  */
 PUSHORT		DataBuffer;

 /*
  * Data words left.
  */
 ULONG			WordsLeft;

 /*
  * Retry count.
  */
 ULONG			RetryCount;

 /*
  * Keep DMA type (MULTIWORD_DMA or ULTRA_DMA) for each device.
  */
 UCHAR			DmaType[2];

 /*
  * Keep UDMA timing for each device.
  */
 UCHAR			UdmaTiming[2];

 /*
  * Keep PIO/DMA timing for each channel. PioDmaTiming[clock][channel]
  */
 UCHAR			PioDmaTiming[2];

 /*
  * Keep IDE clock (50 MHz or 66 MHz) for each device.
  */
 UCHAR			IdeClock[2];

 /*
  * Keep the active device for each channel.
  */
 UCHAR			ActiveDevice;

 /*
  * Indicate whether we should perform DMA mode switch on this channel?
  */
 UCHAR			DoSwitch;

 /*
  * ???
  */
 BOOLEAN		ConvertCdb;

 /*
  * Use or do not use DMA.
  */
 BOOLEAN		UseDma[2];

 /*
  * Flags for initialization.
  */
 BOOLEAN		Initializing;

 /*
  * Reserved for alignment.
  */
 UCHAR			reserved[3];

 /*
  * Identify data for device.
  */
 IDENTIFY_DATA		FullIdentifyData;
 IDENTIFY_DATA2		IdentifyData[2];

 /*
  * DMA PRD table physical address.
  */
 dma_addr_t		dmatable_dma;

 /*
  * DMA PRD table virtual address.
  */
 unsigned int *		dmatable_cpu;

 /*
  * Point to SCATTER/GATHER data buffer.
  */
 struct scatterlist *	sg_table;

 /*
  * How many SCATTER/GATHER counts.
  */
 int			sg_nents;

 /*
  * DMA read or write.
  */
 int			sg_dma_direction;

 /*
  * For retry timer.
  */
 struct timer_list *	retry_timer;

 /*
  * For watchdog timer.
  */
 struct timer_list *	watch_timer;

 /*
  * Current request on controller.
  */
 PSCSI_REQUEST_BLOCK	CurrentSrb;

 /*
  * Original request on controller.
  */
 PSCSI_REQUEST_BLOCK	OriginalSrb;

 /*
  * Internal SRB.
  */
 SCSI_REQUEST_BLOCK	_Srb;

 /*
  * Remember the PCI device.
  */
 struct pci_dev *	pPciDev;

 /*
  * Placeholder for CDB.
  */
 UCHAR	TempCdb[MAXIMUM_CDB_SIZE];
} Channel, *PChannel;

/************************************************************************
 * _Adapter
 ************************************************************************/
typedef struct _Adapter
{
 char *			name;		/* Adapter's name		*/
 UCHAR			num_channels;	/* How many channels support	*/
 UCHAR			irq;		/* irq number			*/
 UCHAR			irqOwned;	/* If any irq is use		*/
 UCHAR			pci_bus;	/* PCI bus number		*/	
 UCHAR			devfn;		/* Device and function number	*/
 UCHAR			offline;	/* On line or off line		*/
 BOOLEAN		bypass_mode;	/* bypass or firware mode	*/	
 UCHAR			reserved2[1];	/* Reserved for alignment	*/
 Channel *		IDEChannel;	/* IT8212 supports two channels	*/
 struct pci_dev *	pci_dev;	/* For PCI device		*/
}
ITE_ADAPTER, *PITE_ADAPTER;

/************************************************************************
 * Function prototypes
 ************************************************************************/
const char *iteraid_info (struct Scsi_Host *);
int iteraid_detect (Scsi_Host_Template *);
int iteraid_release (struct Scsi_Host *);
int iteraid_command (Scsi_Cmnd *);
int iteraid_old_abort (Scsi_Cmnd *);
int iteraid_old_reset (Scsi_Cmnd *, unsigned int);
int iteraid_queuecommand (Scsi_Cmnd *, void (*done)(Scsi_Cmnd *));
int iteraid_biosparam (Disk *, kdev_t, int *);
int iteraid_proc_info (char *buffer, char **start, off_t offset,
			int length, int hostno, int inout);

/************************************************************************
 * Scsi_Host_Template Initializer
 ************************************************************************/
#define ITERAID								\
{									\
 next:				NULL,					\
 module:			NULL,					\
 proc_name:			"it8212",				\
 proc_info:			iteraid_proc_info,			\
 name:				"RAIDExpress133",			\
 detect:			iteraid_detect,				\
 release:			iteraid_release,			\
 info:	     			NULL,					\
 ioctl:				NULL,					\
 command:			iteraid_command,			\
 queuecommand:  		iteraid_queuecommand,			\
 abort:	    			iteraid_old_abort,			\
 reset:	    			iteraid_old_reset,			\
 eh_abort_handler:		iteraid_abort_eh,			\
 eh_device_reset_handler:	NULL,					\
 eh_bus_reset_handler:		NULL,					\
 eh_host_reset_handler:		iteraid_reset_eh,			\
 slave_attach:			NULL,					\
 bios_param:			iteraid_biosparam,			\
 can_queue:			1,					\
 this_id:			-1,					\
 sg_tablesize:   		16,					\
 cmd_per_lun:			32,					\
 present:			0,					\
 unchecked_isa_dma:		0,					\
 use_clustering:   		ENABLE_CLUSTERING,			\
 use_new_eh_code:		1					\
}

/************************************************************************
 * Beautification macros
 ************************************************************************/
#define MAX_RETRY_COUNT		5

#define ScheduleRetryProcess(pChan) do {		\
	pChan->retry_timer->expires = jiffies + 10;	\
	add_timer(pChan->retry_timer);			\
	} while (0)

#define CancelRetryProcess(pChan) del_timer(pChan->retry_timer)

#define GetStatus(pChan, Status)	\
    Status = inb(pChan->io_ports[IDE_CONTROL_OFFSET]);

#define GetBaseStatus(pChan, Status)	\
    Status = inb(pChan->io_ports[IDE_COMMAND_OFFSET]);

#define GetError(pChan, Error)		\
    Error = inb(pChan->io_ports[IDE_ERROR_OFFSET]);

#define ReadBuffer(pChan, Buffer, Count) \
    insw(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);

#define WriteCommand(BaseIoAddress, Command) \
    outb(pChan->io_ports[IDE_COMMAND_OFFSET], Command);

#define WriteBuffer(pChan, Buffer, Count) \
    outsw(pChan->io_ports[IDE_DATA_OFFSET], Buffer, Count);

#define WaitOnBusy(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	}					\
     else 					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitOnBaseBusy(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 20000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(150);				\
	 continue;				\
	} 					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitForDrq(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 1000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
}

#define WaitForBaseDrq(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(200);				\
	}					\
    }						\
}

#define CheckBusyDrq(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetBaseStatus(pChan, Status);		\
     if ((Status & IDE_STATUS_BUSY) ||		\
	!(Status & IDE_STATUS_DRQ))		\
	{					\
	 udelay(200);				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitShortForDrq(pChan, Status)		\
{						\
 ULONG		i;				\
 for (i = 0; i < 2; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status & IDE_STATUS_BUSY)		\
	{					\
	 udelay(100);				\
	}					\
     else if (Status & IDE_STATUS_DRQ)		\
	{					\
	 break;					\
	}					\
     else					\
	{					\
	 udelay(100);				\
	}					\
    }						\
}

#define WaitForDeviceReady(pChan, Status)	\
{						\
 ULONG		i;				\
 for (i = 0; i < 50000; i++)			\
    {						\
     GetStatus(pChan, Status);			\
     if (Status == 0)				\
	{					\
	 break;					\
	}					\
     if ((Status & IDE_STATUS_BUSY) || (Status & IDE_STATUS_DRQ)) \
	{					\
	 udelay(200);				\
	 continue;				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
}

#define WaitForCommandComplete(pChan, Status)		\
{							\
 ULONG		i;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetStatus(pChan, Status);				\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE))			\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
}

#define WaitForBaseCommandComplete(pChan, Status)	\
{							\
 ULONG		i;					\
 for (i = 0; i < 50000; i++)				\
    { 							\
     GetBaseStatus(pChan, Status);			\
     if ((Status == 0) || (Status & IDE_STATUS_ERROR)	\
      || (Status == IDE_STATUS_IDLE))			\
	{						\
	 break;						\
	}						\
     udelay(200);					\
     continue;						\
    }							\
}

#define AtapiSoftReset(pChan, DevNum)		\
{						\
 UCHAR		statusByte;			\
 outb((UCHAR)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 udelay(500);									  \
 outb(IDE_COMMAND_ATAPI_RESET, pChan->io_ports[IDE_COMMAND_OFFSET]);		  \
 mdelay(1000);									  \
 outb((UCHAR)(((DevNum & 0x1) << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]); \
 WaitOnBusy(pChan, statusByte);			\
 udelay(500);					\
}

#define IdeHardReset(pChan, result)		\
{						\
 UCHAR		statusByte;			\
 ULONG		i;				\
 outb(IDE_DC_RESET_CONTROLLER, pChan->io_ports[IDE_CONTROL_OFFSET]);	\
 mdelay(50);								\
 outb(IDE_DC_REENABLE_CONTROLLER, pChan->io_ports[IDE_CONTROL_OFFSET]);	\
 for (i = 0; i < 1000 * 1000; i++)					\
    {									\
     statusByte = inb(pChan->io_ports[IDE_CONTROL_OFFSET]);		\
     if (statusByte != IDE_STATUS_IDLE && statusByte != 0x0)		\
	{					\
	 udelay(30);				\
	}					\
     else					\
	{					\
	 break;					\
	}					\
    }						\
 if (i == 1000 * 1000)				\
    {						\
     printk("IdeHardReset Fail!\n");		\
     result = FALSE;				\
    }						\
 else						\
    {						\
     dprintk("IdeHardReset Success!\n");	\
     result = TRUE;				\
    }						\
}

/************************************************************************
 * Function prototypes
 ************************************************************************/
void		iteretry(PChannel);
void		TaskStart(PChannel, Scsi_Cmnd *);
void		TaskQueue(void);
void		TaskDone(PChannel, PSCSI_REQUEST_BLOCK);
ULONG		IdeSendCommand(PChannel, PSCSI_REQUEST_BLOCK);
void		IdeMediaStatus(BOOLEAN, PChannel, ULONG);
static void	IdeSetupDma(PChannel, ULONG, USHORT);
void		MapRequest(Scsi_Cmnd *, PSCSI_REQUEST_BLOCK);
BOOLEAN		IssueIdentify(PChannel, UCHAR, UCHAR);
BOOLEAN		AtapiStartIo(PChannel, PSCSI_REQUEST_BLOCK);
BOOLEAN		IT8212ResetAdapter(void);
BOOLEAN		AtapiInterrupt(PChannel);

static int	itedev_open(struct inode *, struct file *);
static int	itedev_ioctl_entry(struct inode *, struct file *, unsigned int, unsigned long);
static int	itedev_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static int	itedev_close(struct inode *, struct file *);

#endif /* #ifndef _ITERAID_H_ */
