/*
  Copyright (c) 1996,2003 VIA Networking, Inc. All rights reserved.

  This software may be redistributed and/or modified
  under the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2 of the License, or
  any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

*/
 
#undef __NO_VERSION__

#include "rhine.h"

static int			rhine_nics				=0;
static PRHINE_INFO	pRhine3_Infos			=NULL;
static int			msglevel				=MSG_LEVEL_INFO;

#ifdef	RHINE_ETHTOOL_IOCTL_SUPPORT
static int 	rhine_ethtool_ioctl(struct net_device* dev, struct ifreq* ifr);
#endif

#ifdef SIOCGMIIPHY
static int 	rhine_mii_ioctl(struct net_device* dev, struct ifreq* ifr, int cmd);
#endif


/*
	Define module options
*/

MODULE_AUTHOR("VIA Technologies, Inc., <ShingChuang@via.com.tw>");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_DESCRIPTION("VIA Rhine Farmily Fast Ethernet Adapter Driver");

#define RHINE_PARAM(N,D) \
		static const int N[MAX_UINTS]=OPTION_DEFAULT;\
		MODULE_PARM(N, "1-" __MODULE_STRING(MAX_UINTS) "i");\
		MODULE_PARM_DESC(N, D);

#define RX_DESC_MIN		16
#define RX_DESC_MAX		128
#define RX_DESC_DEF		64
RHINE_PARAM(RxDescriptors,"Number of receive descriptors");

#define TX_DESC_MIN		16
#define TX_DESC_MAX		128
#define TX_DESC_DEF		64
RHINE_PARAM(TxDescriptors,"Number of transmit descriptors");

#define VLAN_ID_MIN		0
#define VLAN_ID_MAX		4095
#define VLAN_ID_DEF		0
/* VID_setting[] is used for setting the VID of NIC.
   0: default VID.
   1-4094: other VIDs.
*/
RHINE_PARAM(VID_setting,"802.1q VLAN ID");

#define TX_THRESH_MIN	0
#define TX_THRESH_MAX	4
#define TX_THRESH_DEF	0
/* tx_thresh[] is used for controlling the transmit fifo threshold.
   0: indicate the txfifo threshold is 128 bytes.
   1: indicate the txfifo threshold is 256 bytes.
   2: indicate the txfifo threshold is 512 bytes.
   3: indicate the txfifo threshold is 1024 bytes.
   4: indicate that we use store and forward
*/
RHINE_PARAM(tx_thresh,"Transmit fifo threshold");

#define RX_THRESH_MIN	0
#define RX_THRESH_MAX	7
#define RX_THRESH_DEF	0
/* rx_thresh[] is used for controlling the receive fifo threshold.
   0: indicate the rxfifo threshold is 64 bytes.
   1: indicate the rxfifo threshold is 32 bytes.
   2: indicate the rxfifo threshold is 128 bytes.
   3: indicate the rxfifo threshold is 256 bytes.
   4: indicate the rxfifo threshold is 512 bytes.
   5: indicate the rxfifo threshold is 768 bytes.
   6: indicate the rxfifo threshold is 1024 bytes.   
   7: indicate that we use store and forward
*/
RHINE_PARAM(rx_thresh,"Receive fifo threshold");

#define DMA_LENGTH_MIN	0
#define DMA_LENGTH_MAX	7
#define DMA_LENGTH_DEF	0
/* DMA_length[] is used for controlling the DMA length
   0: 8 DWORDs
   1: 16 DWORDs
   2: 32 DWORDs
   3: 64 DWORDs
   4: 128 DWORDs
   5: 256 DWORDs
   6: SF(flush till emply)
   7: SF(flush till emply)
*/
RHINE_PARAM(DMA_length,"DMA length");

#define TAGGING_DEF		0
/* enable_tagging[] is used for enabling 802.1q VID tagging.
   0: disable VID seeting(default).
   1: enable VID setting.
*/
RHINE_PARAM(enable_tagging,"Enable 802.1Q tagging");

#define	IP_ALIG_DEF		0
/* IP_byte_align[] is used for IP header DWORD byte aligned
   0: indicate the IP header won't be DWORD byte aligned.(Default) .
   1: indicate the IP header will be DWORD byte aligned.
      In some enviroment, the IP header should be DWORD byte aligned,
      or the packet will be droped when we receive it. (eg: IPVS)
*/
RHINE_PARAM(IP_byte_align,"Enable IP header dword aligned");

#define RX_CSUM_DEF		1
/* rxcsum_offload[] is used for setting the checksum offload ability of NIC.
   (We only support RX checksum offload now)
   0: disable csum_offload[checksum offload
   1: enable checksum offload. (Default)
*/
RHINE_PARAM(rxcsum_offload,"Enable receive packet checksum offload");

#ifdef RHINE_TX_CSUM_SUPPORT
#define TX_CSUM_DEF		0
/* txcsum_offload[] is used for setting the checksum offload ability of NIC.
   (We only support RX checksum offload now)
   0: disable csum_offload[checksum offload
   1: enable checksum offload. (Default)
*/
RHINE_PARAM(txcsum_offload,"Enable transmit packet checksum offload");
#endif

#define FLOW_CNTL_DEF	1
#define FLOW_CNTL_MIN	1
#define	FLOW_CNTL_MAX	3
/* flow_control[] is used for setting the flow control ability of NIC.
   1: hardware deafult(default). Use Hardware default value in ANAR.
   2: disable PAUSE in ANAR.
   3: enable PAUSE in ANAR.
*/
RHINE_PARAM(flow_control,"Enable flow control ability");

#define MED_LNK_DEF	0
#define MED_LNK_MIN 0
#define MED_LNK_MAX 4
/* speed_duplex[] is used for setting the speed and duplex mode of NIC.
   0: indicate autonegotiation for both speed and duplex mode
   1: indicate 100Mbps half duplex mode
   2: indicate 100Mbps full duplex mode
   3: indicate 10Mbps half duplex mode
   4: indicate 10Mbps full duplex mode
   
   Note:
   		if EEPROM have been set to the force mode, this option is ignored
   			by driver.
*/
RHINE_PARAM(speed_duplex,"Setting the speed and duplex mode");

#define VAL_PKT_LEN_DEF		0
/* ValPktLen[] is used for setting the checksum offload ability of NIC.
   0: Receive frame with invalid layer 2 length (Default)
   1: Drop frame with invalid layer 2 length
*/
RHINE_PARAM(ValPktLen,"Receiving or Drop invalid 802.3 frame");

#define WOL_OPT_DEF		0
#define WOL_OPT_MIN		0
#define WOL_OPT_MAX		7
/* wol_opts[] is used for controlling wake on lan behavior.
   0: Wake up if recevied a magic packet. (Default)
   1: Wake up if link status is on/off.
   2: Wake up if recevied an arp packet.
   4: Wake up if recevied any unicast packet.
   Those value can be sumed up to support more than one option.
*/
RHINE_PARAM(wol_opts,"Wake On Lan options");

#define INT_WORKS_DEF	20
#define	INT_WORKS_MIN	10
#define INT_WORKS_MAX	64

RHINE_PARAM(int_works,"Number of packets per interrupt services");

static int 	rhine_found1(struct pci_dev *pcid, const struct pci_device_id *ent);
static BOOL	rhine_init_info(struct pci_dev* pcid, PRHINE_INFO* ppInfo, PCHIP_INFO);
static void rhine_free_info(PRHINE_INFO pInfo);
static BOOL	rhine_get_pci_info(PRHINE_INFO, struct pci_dev* pcid);
static void	rhine_print_info(PRHINE_INFO pInfo);
static int  rhine_open(struct net_device *dev);
static int  rhine_xmit(struct sk_buff *skb, struct net_device *dev);
static void rhine_intr(int irq, void *dev_instance, struct pt_regs *regs);
static void rhine_set_multi(struct net_device *dev);
static struct net_device_stats *rhine_get_stats(struct net_device *dev);
static int	rhine_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int  rhine_close(struct net_device *dev);
static int	rhine_rx_srv(PRHINE_INFO pInfo,int status);
static BOOL	rhine_receive_frame(PRHINE_INFO ,PRX_DESC pDesc);
static BOOL rhine_alloc_rx_buf(PRHINE_INFO pInfo, PRX_DESC pDesc);
static void rhine_init_registers(PRHINE_INFO pInfo, RHINE_INIT_TYPE);
static void rhine_free_tx_buf(PRHINE_INFO pInfo, PTX_DESC pDesc);
static BOOL rhine_soft_reset(PRHINE_INFO pInfo);
static void mii_get_phy_id(PRHINE_INFO pInfo);
static BOOL rhine_mii_read(PRHINE_INFO pInfo, U8 byIdx, PU16 pdata);
static BOOL rhine_mii_write (PRHINE_INFO pInfo, BYTE byMiiAddr, WORD wData);
static U32 rhine_get_opt_media_mode(PRHINE_INFO pInfo);
static void rhine_print_link_status(PRHINE_INFO pInfo);
static void SafeDisableMiiAutoPoll (PRHINE_INFO pInfo);
static void rhine_shutdown(PRHINE_INFO pInfo);
static void enable_flow_control_ability(PRHINE_INFO pInfo);
static void  EnableMiiAutoPoll(PMAC_REGS pMacRegs);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
static int rhine_notify_reboot(struct notifier_block *, unsigned long event, void *ptr);
static int rhine_suspend(struct pci_dev *pcid, u32 state);
static int rhine_resume(struct pci_dev *pcid);
struct notifier_block rhine_notifier = {
        notifier_call:  rhine_notify_reboot,
        next:           NULL,
        priority:       0
};

static int 
rhine_netdev_event(struct notifier_block *nb,
	unsigned long notification, void *ptr);

static struct notifier_block rhine_inetaddr_notifier = {
    notifier_call: rhine_netdev_event,
    };
    
#endif
#endif 


static CHIP_INFO chip_info_table[]= {
	{ VT86C100A, 	"VIA VT86C100A Rhine I Fast Ethernet Adapter                 ",
		128, 1, 	RHINE_FLAGS_IP_ALIGN|RHINE_FLAGS_TX_ALIGN},
	{ VT6102,		"VIA VT6102 Rhine II Fast Ethernet Adapter                   ",
		256, 1,		RHINE_FLAGS_IP_ALIGN },
	{ VT6105, 		"VIA VT6105 Rhine III Fast Ethernet Adapter                  ",
		256, 1, 	RHINE_FLAGS_IP_ALIGN|RHINE_FLAGS_FLOW_CTRL},
	{ VT6105M, 		"VIA VT6105M Rhine III Management Adapter                    ",
		256, 1,		RHINE_FLAGS_IP_ALIGN|RHINE_FLAGS_FLOW_CTRL|
					RHINE_FLAGS_TAGGING|	RHINE_FLAGS_TX_CSUM|	
					RHINE_FLAGS_HAVE_CAM|	RHINE_FLAGS_RX_CSUM},
	{0,NULL}
};

static struct pci_device_id rhine_id_table[] __devinitdata = {
{ 0x1106, 0x3043, PCI_ANY_ID, PCI_ANY_ID, 0, 0, (int)&chip_info_table[0]},
{ 0x1106, 0x3065, PCI_ANY_ID, PCI_ANY_ID, 0, 0, (int)&chip_info_table[1]},
{ 0x1106, 0x3106, PCI_ANY_ID, PCI_ANY_ID, 0, 0, (int)&chip_info_table[2]},
{ 0x1106, 0x3053, PCI_ANY_ID, PCI_ANY_ID, 0, 0, (int)&chip_info_table[3]},
{ 0, }
};

static char* get_chip_name(int chip_id) {
	int i;
	for (i=0;chip_info_table[i].name!=NULL;i++)
		if (chip_info_table[i].chip_id==chip_id)
			break;
	return chip_info_table[i].name;	
}

static void __devexit rhine_remove1(struct pci_dev *pcid)
{
	PRHINE_INFO pInfo=pci_get_drvdata(pcid);
	
	if (pInfo==NULL)
		return;
		
	if (pInfo->flags & RHINE_FLAGS_WOL_ENABLED) {
		rhine_get_ip(pInfo);
		rhine_set_wol(pInfo);
		pci_enable_wake(pcid,3,1);
		pci_set_power_state(pcid,1);
	}
		
#ifdef CONFIG_PROC_FS
	rhine_free_proc_entry(pInfo);	
#endif
	
	rhine_free_info(pInfo);

}

static void 
rhine_set_int_opt(int *opt, int val, int min, int max, int def,char* name,char* devname) {
	if (val==-1)
		*opt=def;
	else if (val<min || val>max) {
		RHINE_PRT(MSG_LEVEL_INFO, KERN_NOTICE "%s: the value of parameter %s is invalid, the valid range is (%d-%d)\n" ,
			devname,name, min,max);
		*opt=def;
	} else {
		RHINE_PRT(MSG_LEVEL_INFO, KERN_INFO "%s: set value of parameter %s to %d\n",
			devname, name, val);
		*opt=val;
	}
}

static void 
rhine_set_bool_opt(PU32 opt, int val,BOOL def,U32 flag, char* name,char* devname) {
	(*opt)&=(~flag);	
	if (val==-1) 
		*opt|=(def ? flag : 0);
	else if (val<0 || val>1) {
		printk(KERN_NOTICE
			"%s: the value of parameter %s is invalid, the valid range is (0-1)\n",devname,name);
		*opt|=(def ? flag : 0);
	} else {
		printk(KERN_INFO "%s: set parameter %s to %s\n",
			devname,name , val ? "TRUE" : "FALSE");			
		*opt|=(val ? flag : 0);
	}
}

static void
rhine_get_options(POPTIONS pOpts,int index,char* devname) {

	rhine_set_int_opt(&pOpts->tx_thresh,tx_thresh[index],
		TX_THRESH_MIN, TX_THRESH_MAX, RX_THRESH_DEF,"tx_thresh",devname);
	
	rhine_set_int_opt(&pOpts->rx_thresh,rx_thresh[index],
		RX_THRESH_MIN, RX_THRESH_MAX, RX_THRESH_DEF,"rx_thresh",devname);
	
	rhine_set_int_opt(&pOpts->DMA_length,DMA_length[index],
		DMA_LENGTH_MIN, DMA_LENGTH_MAX, DMA_LENGTH_DEF,"DMA_length",devname);
	
	rhine_set_int_opt(&pOpts->nRxDescs,RxDescriptors[index],
		RX_DESC_MIN, RX_DESC_MAX, RX_DESC_DEF, "RxDescriptors",devname);
	
	rhine_set_int_opt(&pOpts->nTxDescs,TxDescriptors[index],
		TX_DESC_MIN, TX_DESC_MAX, TX_DESC_DEF, "TxDescriptors",devname);

	rhine_set_int_opt(&pOpts->vid,VID_setting[index],
		VLAN_ID_MIN, VLAN_ID_MAX, VLAN_ID_DEF,"VID_setting",devname);	

	rhine_set_bool_opt(&pOpts->flags,enable_tagging[index],
		TAGGING_DEF,RHINE_FLAGS_TAGGING, "enable_tagging",devname);
	
	rhine_set_bool_opt(&pOpts->flags,rxcsum_offload[index],
		RX_CSUM_DEF,RHINE_FLAGS_RX_CSUM,"rxcsum_offload",devname);
		
#ifdef RHINE_TX_CSUM_SUPPORT	
	rhine_set_bool_opt(&pOpts->flags,txcsum_offload[index],
		TX_CSUM_DEF,RHINE_FLAGS_TX_CSUM,"txcsum_offload",devname);
#endif
	
	rhine_set_int_opt(&pOpts->flow_cntl,flow_control[index],
		FLOW_CNTL_MIN,FLOW_CNTL_MAX, FLOW_CNTL_DEF, "flow_control",devname);
	
	rhine_set_bool_opt(&pOpts->flags,IP_byte_align[index],
		IP_ALIG_DEF, RHINE_FLAGS_IP_ALIGN, "IP_byte_align",devname);
	
	rhine_set_bool_opt(&pOpts->flags,ValPktLen[index],
		VAL_PKT_LEN_DEF, RHINE_FLAGS_IP_ALIGN, "ValPktLen",devname);
		
	rhine_set_int_opt((int*) &pOpts->spd_dpx,speed_duplex[index],
		MED_LNK_MIN, MED_LNK_MAX, MED_LNK_DEF,"Media link mode",devname);	
			
	rhine_set_int_opt((int*) &pOpts->wol_opts,wol_opts[index],
		WOL_OPT_MIN, WOL_OPT_MAX, WOL_OPT_DEF,"Wake On Lan options",devname);	
		
	rhine_set_int_opt((int*) &pOpts->int_works,int_works[index],
		INT_WORKS_MIN, INT_WORKS_MAX, INT_WORKS_DEF,"Interrupt service works",devname);
				
}

/*
static void rhine_print_options(PRHINE_INFO pInfo) {
	if (pInfo->flags & RHINE_FLAGS_TAGGING)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: Hardware 802.1q tagging enabled (VID=%d)\n",
				pInfo->dev->name, pInfo->sOpts.vid);
			
	if (pInfo->flags & RHINE_FLAGS_TX_CSUM)			
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: Hardware TX checksum offload enabled\n",
				pInfo->dev->name);	
			
	if (pInfo->flags & RHINE_FLAGS_RX_CSUM)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: Hardware RX checksum offload enabled\n",
				pInfo->dev->name);			
			
	if (pInfo->flags & RHINE_FLAGS_FLOW_CTRL)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: Hardware flow control enabled\n",
				pInfo->dev->name);
				
	if (pInfo->flags & RHINE_FLAGS_HAVE_CAM)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: Hardware multicast cam enabled\n",
				pInfo->dev->name);	
				
	if (pInfo->flags & RHINE_FLAGS_TX_ALIGN)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: TX buffer dword alignment enabled\n",
				pInfo->dev->name);
				
	if (pInfo->flags & RHINE_FLAGS_IP_ALIGN)
		RHINE_PRT(MSG_LEVEL_INFO, 
			KERN_INFO "%s: OPTION: RX buffer IP header dword alignment enabled\n",
				pInfo->dev->name);
				
}
*/
void rhine_init_cam_filter(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	
	BYTE_REG_BITS_SET((TCR_PQEN),(TCR_RTGOPT|TCR_PQEN),&pMacRegs->byTCR);
	BYTE_REG_BITS_ON(BCR1_VIDFR,&pMacRegs->byBCR1);
	//Disable all cam
	rhine_set_cam_mask(pInfo,0,RHINE_VLAN_ID_CAM);
	rhine_set_cam_mask(pInfo,0,RHINE_MULTICAST_CAM);		
	//Enable first vcam 
	if (pInfo->flags & RHINE_FLAGS_TAGGING) {
		rhine_set_cam(pInfo,0,(PU8) &(pInfo->sOpts.vid),RHINE_VLAN_ID_CAM);
		rhine_set_cam_mask(pInfo,1,RHINE_VLAN_ID_CAM);
		rhine_set_cam_mask(pInfo,0,RHINE_MULTICAST_CAM);
	} 

}
//
// Initialiation of MAC registers
//
static void 
rhine_init_registers(PRHINE_INFO pInfo, RHINE_INIT_TYPE InitType) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	int 		i,mii_status;
	U8			byRevId=pInfo->byRevId;

	rhine_wol_reset(pInfo);
	switch (InitType) {		
	case RHINE_INIT_RESET:
	case RHINE_INIT_WOL:

		netif_stop_queue(pInfo->dev);
		mii_status=rhine_get_opt_media_mode(pInfo);		
		if (rhine_set_media_mode(pInfo,mii_status)!=RHINE_LINK_CHANGE) {
			rhine_print_link_status(pInfo);
			if (!(pInfo->mii_status & RHINE_LINK_FAIL))
				netif_wake_queue(pInfo->dev);	
		} 
		else if (pInfo->mii_status & RHINE_LINK_FAIL) 
			rhine_print_link_status(pInfo);
				
		if (byRevId>=REV_ID_VT6102_A)
			enable_flow_control_ability(pInfo);
				
		rhine_ClearISR(pInfo->pMacRegs);
	
		writew((CR0_DPOLL|CR0_TXON|CR0_RXON|CR0_STRT),&pMacRegs->byCR0);
		return;
		break;
	
	case RHINE_INIT_COLD:	
	default:


	    // do reset
	    rhine_soft_reset(pInfo);
		mdelay(5);
	    		
		reload_eeprom(pInfo);
		
		for (i = 0; i < 6; i++)
			writeb(pInfo->dev->dev_addr[i], &pMacRegs->abyPAR[i]);		
			
		rhine_set_tx_thresh(pInfo,pInfo->sOpts.tx_thresh);
		rhine_set_rx_thresh(pInfo,pInfo->sOpts.rx_thresh);
		rhine_set_DMA_length(pInfo,pInfo->sOpts.DMA_length);
	
	    // for VT3043,VT3071 only
	    if (byRevId < REV_ID_VT6102_A) {
        // disable queue packet, PATCH.... Aladdin 4/5 bug
    	    BYTE_REG_BITS_ON(CFGB_QPKTDIS, &pMacRegs->byCFGB);
	    }
	    else {		
        	// enable queue packet
    	    BYTE_REG_BITS_OFF(CFGB_QPKTDIS, &pMacRegs->byCFGB);
	        // suspend-well accept broadcast, multicast
    	    writeb(WOLCFG_SAM|WOLCFG_SAB, &pMacRegs->byWOLCGSet);
	    }

	    // back off algorithm use original IEEE standard
	    BYTE_REG_BITS_ON(TCR_OFSET,&pMacRegs->byTCR);
		BYTE_REG_BITS_OFF((CFGD_CRADOM | CFGD_CAP | CFGD_MBA | CFGD_BAKOPT),
			&pMacRegs->byCFGD);        

	    // set packet filter
	    // receive directed and broadcast address

		rhine_set_multi(pInfo->dev); 
	
		pInfo->IntMask=
			(IMR_PRXM 	| IMR_PTXM	| IMR_RXEM	| IMR_TXEM	| IMR_TUM	| 
			IMR_RUM		| IMR_BEM 	| IMR_CNTM 		| IMR_ERM 	| IMR_ETM	|
			IMR_ABTM 	| IMR_SRCM	| IMR_NORBFM | IMR_OVFM);
			
		if (pInfo->byRevId>REV_ID_VT6102_A) {
		 	if (pInfo->byRevId<REV_ID_VT6105M_A0)
				pInfo->IntMask|=(IMR_GENM|IMR_TDWBRAI);
			else
				pInfo->IntMask|=(IMR_GENM|IMR_TM1_INT);
		}
		
	
		writel(cpu_to_le32(pInfo->rd_pool_dma),&pMacRegs->dwCurrRxDescAddr);	
	
		for (i=0;i<pInfo->nTxQueues;i++)
			writel(cpu_to_le32(pInfo->td_pool_dma[i]),&(pMacRegs->adwCurrTxDescAddr[i]));

		if (byRevId>=REV_ID_VT6105M_A0)
			rhine_init_cam_filter(pInfo);

		if (pInfo->flags & RHINE_FLAGS_FLOW_CTRL)
			enable_flow_control(pInfo);
	
		writew((CR0_DPOLL|CR0_TXON|CR0_RXON|CR0_STRT),&pMacRegs->byCR0);		
	
		mii_status=rhine_get_opt_media_mode(pInfo);	

		netif_stop_queue(pInfo->dev);
		if (rhine_set_media_mode(pInfo,mii_status)!=RHINE_LINK_CHANGE) {
			rhine_print_link_status(pInfo);
			if (!(pInfo->mii_status & RHINE_LINK_FAIL))
				netif_wake_queue(pInfo->dev);	
		} 
		else if (pInfo->mii_status & RHINE_LINK_FAIL)
			rhine_print_link_status(pInfo);
	
		if (byRevId>=REV_ID_VT6102_A)
			enable_flow_control_ability(pInfo);
				
		rhine_ClearISR(pInfo->pMacRegs);
	}
}

static BOOL rhine_soft_reset(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	int i=0;
			
	BYTE_REG_BITS_ON(CR1_SFRST, &pMacRegs->byCR1);
	
    /* VT86C100A may need long delay after reset (dlink) */
    if (pInfo->chip_id == VT86C100A)
		udelay(100);
                                
    for (i=0;i<W_MAX_TIMEOUT;i++) {
		udelay(5);
		if (!(readb(&pMacRegs->byCR1) & CR1_SFRST)) 
			break;
	}
	
	if (i==W_MAX_TIMEOUT) {
       	if (pInfo->chip_id==VT86C100A) {
	       	printk(KERN_ERR "%s: reset did not complete in 10 ms.\n", 
	       		pInfo->dev->name); 
       		return FALSE;
       	}
       		
        writeb(MISC_CR1_FORSRST, &pMacRegs->byMISCCR1);
        // delay 2ms
       	mdelay(2);
	}
	return TRUE;	
}


static int 
rhine_found1(struct pci_dev *pcid, const struct pci_device_id *ent)
{
	static BOOL			bFirst=TRUE;
	struct net_device*	dev=NULL;
	int					i;
	
	PCHIP_INFO			pChip_info=(PCHIP_INFO) ent->driver_data;
		
	PRHINE_INFO		pInfo;
	PMAC_REGS			pMacRegs;

	if (rhine_nics++>=MAX_UINTS) {
		printk(KERN_NOTICE RHINE_NAME ": already found %d NICs\n", rhine_nics);
		return -ENODEV;
	}
	dev=init_etherdev(dev,0);

	if (dev==NULL) {
		printk(KERN_ERR RHINE_NAME ": allocate net device failed");
		return -ENODEV;
	}

	if (bFirst) {
		printk(KERN_INFO "%s Ver. %s\n",RHINE_FULL_DRV_NAM, RHINE_VERSION);
		printk(KERN_INFO "Copyright (c) 2002 VIA Technologies, Inc.\n");
		bFirst=FALSE;
	}

	if (!rhine_init_info(pcid,&pInfo, pChip_info)) {
		return -ENOMEM;		
	}
	pInfo->dev=dev;
	
	dev->priv=pInfo;	
	dev->irq=pcid->irq;
	
	if (pci_enable_device(pcid)) {
		rhine_free_info(pInfo);
		return -ENODEV;
	}

	if (rhine_get_pci_info(pInfo,pcid)==FALSE) {
		printk(KERN_ERR RHINE_NAME ": Failed to find PCI device\n");	
		rhine_free_info(pInfo);
		return -ENODEV;
	}
	
	rhine_create_proc_entry(pInfo);			
	
	enable_mmio(pInfo);	
	
	pMacRegs=ioremap(pInfo->memaddr & PCI_BASE_ADDRESS_MEM_MASK, pInfo->io_size);
	pInfo->pMacRegs=pMacRegs;		

	mii_get_phy_id(pInfo);
	
	if (pInfo->byRevId>REV_ID_VT86C100A_E) {
        // clear sticky bits
 		BYTE_REG_BITS_OFF((STICKHW_DS1_SHADOW|STICKHW_DS0_SHADOW),
			&pMacRegs->bySTICKHW);			
        // disable force PME-enable 			        
		writeb(WOLCR_LNKOFF_EN, &pMacRegs->byWOLCRClr);
        // disable power-event config bit        
		writeb(0xFF, &pMacRegs->byWOLCGClr);
        // clear power status        				
		writeb(0xFF, &pMacRegs->byPWRCSRClr);
	}	
	
	if (check_region(pInfo->ioaddr,pInfo->io_size)) {
		printk(KERN_ERR RHINE_NAME ": Failed to find PCI device\n");
		pInfo->ioaddr=0;
		rhine_free_info(pInfo);
		return -ENODEV;
	}	

	request_region(pInfo->ioaddr,pInfo->io_size,RHINE_NAME);
		
	//Set back-off algorithm 
	BYTE_REG_BITS_OFF((CFGD_CRADOM | CFGD_CAP | CFGD_MBA | CFGD_BAKOPT),
		&pMacRegs->byCFGD);
		
	dev->base_addr=pInfo->ioaddr;
	
	for (i=0;i<6;i++) 
		dev->dev_addr[i]=readb(&pMacRegs->abyPAR[i]);

	rhine_print_info(pInfo);		
	
	rhine_get_options(&pInfo->sOpts,rhine_nics-1,dev->name);	
	
	//Mask out the options cannot be set to the chip
	pInfo->sOpts.flags &= pChip_info->flags;
			
	//Enable the chip specified capbilities
	pInfo->flags=pInfo->sOpts.flags | (pChip_info->flags & 0xFF000000UL);
	
	if (pInfo->byRevId>=REV_ID_VT6102_A) {	
		pInfo->wol_opts=pInfo->sOpts.wol_opts;	
		pInfo->flags|=RHINE_FLAGS_WOL_ENABLED;	
	}	
	
//	rhine_print_options(pInfo);
	
	dev->irq=pcid->irq;		
    dev->open 				= rhine_open;
	dev->hard_start_xmit	= rhine_xmit;
 	dev->stop				= rhine_close;
	dev->get_stats			= rhine_get_stats;
	dev->set_multicast_list	= rhine_set_multi;
	dev->do_ioctl			= rhine_ioctl;
	
#ifdef	RHINE_ZERO_COPY_SUPPORT
	dev->features			|= NETIF_F_SG;
#endif

#ifdef	RHINE_TX_CSUM_SUPPORT
	if ((pInfo->byRevId>=REV_ID_VT6105M_A0)
		&& (pInfo->flags & RHINE_FLAGS_TX_CSUM)) {
		dev->features			|= NETIF_F_IP_CSUM;
	}
#endif	

	pci_set_drvdata(pcid,pInfo);
	
	return 0;
}

static void	rhine_print_info(PRHINE_INFO pInfo)
{
	struct net_device* dev=pInfo->dev;
	
	printk(KERN_INFO "%s: %s\n",dev->name, get_chip_name(pInfo->chip_id));
	printk(KERN_INFO "%s: MAC=%2.2X:%2.2X:%2.2X:%2.2X:%2.2X:%2.2X",
		dev->name,
		dev->dev_addr[0],dev->dev_addr[1],dev->dev_addr[2],
		dev->dev_addr[3],dev->dev_addr[4],dev->dev_addr[5]);
				
	printk(" IO=0x%lx Mem=0x%lx ",(ULONG) pInfo->ioaddr,(ULONG) pInfo->pMacRegs);
	printk(" IRQ=%d \n", pInfo->dev->irq);
			
}

static BOOL	rhine_init_info(struct pci_dev* pcid, PRHINE_INFO* ppInfo,
	PCHIP_INFO pChip_info) {
	PRHINE_INFO	p;
		
	*ppInfo=kmalloc(sizeof(RHINE_INFO),GFP_ATOMIC);
	
	if (*ppInfo==NULL)
		return FALSE;
		
	memset(*ppInfo,0,sizeof(RHINE_INFO));
	
	if (pRhine3_Infos==NULL) {
		pRhine3_Infos=*ppInfo;		
	}
	else {
		for (p=pRhine3_Infos;p->next!=NULL;p=p->next)
			do {} while (0);	
		p->next=*ppInfo;
		(*ppInfo)->prev=p;
	}

	(*ppInfo)->pcid=pcid;
	(*ppInfo)->chip_id=pChip_info->chip_id;
	(*ppInfo)->io_size=pChip_info->io_size;
	(*ppInfo)->nTxQueues=pChip_info->nTxQueue;
	(*ppInfo)->multicast_limit=32;

#ifdef VMNS
	(*ppInfo)->vmns_priv=kmalloc(sizeof(VMNS_DRV_PRIVATE),GFP_ATOMIC);
	if ((*ppInfo)->vmns_priv==NULL)
		return FALSE;
	memset((*ppInfo)->vmns_priv,0,sizeof(VMNS_DRV_PRIVATE));
#endif

	spin_lock_init(&((*ppInfo)->lock));
	spin_lock_init(&((*ppInfo)->xmit_lock));
#ifdef CONFIG_PROC_FS
	rhine_init_proc_fs(*ppInfo);
#endif
	
	return TRUE;
}

static BOOL	rhine_get_pci_info(PRHINE_INFO pInfo, struct pci_dev* pcid) {

	U16	pci_cmd;
    pci_read_config_byte(pcid, PCI_REVISION_ID, &pInfo->byRevId);
    pci_read_config_word(pcid, PCI_SUBSYSTEM_ID,&pInfo->SubSystemID);
    pci_read_config_word(pcid, PCI_SUBSYSTEM_VENDOR_ID, &pInfo->SubVendorID);
    pci_read_config_word(pcid, PCI_COMMAND, (u16 *) & (pci_cmd));
    
    pci_set_master(pcid);
    
   	pInfo->ioaddr=pci_resource_start(pcid,0);    
    pInfo->memaddr=pci_resource_start(pcid,1);
		
	pInfo->pcid=pcid;

	if (pInfo->byRevId>=REV_ID_VT6102_A) {
		U8	b;
		pci_read_config_byte(pcid,0x53, &b);
		pci_write_config_byte(pcid,0x53,b|0x04);
	}
	
	if (pInfo->byRevId>=REV_ID_VT6105_LOM && 
		pInfo->byRevId< REV_ID_VT6105M_A0) {
		U8 b;
		pci_read_config_byte(pcid,0x52, &b);
		pci_write_config_byte(pcid,0x52,b|0x02);
	}
			
	return TRUE;                
}

static void	rhine_free_info(PRHINE_INFO pInfo) {
	PRHINE_INFO			ptr;
	struct net_device*	dev=pInfo->dev;

	ASSERT(pInfo);

	if (pRhine3_Infos==NULL)
		return;
	
	for (ptr=pRhine3_Infos;ptr && (ptr!=pInfo);ptr=ptr->next)
			do {} while (0);
	
	if (ptr==pInfo) {
		if (ptr==pRhine3_Infos)
			pRhine3_Infos=ptr->next;
		else 
			ptr->prev->next=ptr->next; 
	}
	else {
		RHINE_PRT(MSG_LEVEL_ERR, KERN_ERR "vmns: info struct not found\n");
		return;
	}

#ifdef CONFIG_PROC_FS
	rhine_free_proc_fs(pInfo);	
#endif

	if (dev)
		unregister_netdev(dev);
	
	if (pInfo->pMacRegs)
		iounmap(pInfo->pMacRegs);
	
	if (pInfo->ioaddr)
		release_region(pInfo->ioaddr,pInfo->io_size);

	if (dev)		
		kfree(dev);
	
	if (pInfo->pcid) {
		pci_set_drvdata(pInfo->pcid,NULL);		
	}
				
#ifdef VMNS	
	if (pInfo->vmns_priv)
		kfree(pInfo->vmns_priv);
#endif		

	kfree(pInfo);
	
}

static BOOL rhine_init_rings(PRHINE_INFO pInfo) {
	void*	vir_pool;
	int		i;

	/*allocate all RD/TD rings a single pool*/		
	vir_pool=pci_alloc_consistent(pInfo->pcid, 
				    pInfo->sOpts.nRxDescs * sizeof(RX_DESC) +
				    pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
				    &pInfo->pool_dma);
				    
	if (vir_pool==NULL)	{
		printk(KERN_ERR "%s : allocate dma memory failed\n", pInfo->dev->name);
		return FALSE;
	}
	
	memset(vir_pool,0, pInfo->sOpts.nRxDescs * sizeof(RX_DESC) +
	    pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues);
		
	pInfo->aRDRing=vir_pool;
	pInfo->rd_pool_dma=pInfo->pool_dma;
	
	pInfo->tx_bufs=pci_alloc_consistent(pInfo->pcid, 
			    pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues,
			    &pInfo->tx_bufs_dma);		    
		    
	if (pInfo->tx_bufs==NULL) {
		printk(KERN_ERR "%s: allocate dma memory failed\n", pInfo->dev->name);
		pci_free_consistent(pInfo->pcid,
	    	pInfo->sOpts.nRxDescs * sizeof(RX_DESC) +
		    pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
	   		pInfo->aRDRing, pInfo->pool_dma);
		return FALSE;
	}	
	
	memset(pInfo->tx_bufs,0,pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues);

	for (i=0;i<pInfo->nTxQueues;i++) {	

		pInfo->td_pool_dma[i]=pInfo->rd_pool_dma+
				pInfo->sOpts.nRxDescs*sizeof(RX_DESC)+
				pInfo->sOpts.nTxDescs*sizeof(TX_DESC)*i;
				
		pInfo->apTDRings[i]=vir_pool+
				pInfo->sOpts.nRxDescs*sizeof(RX_DESC)+
				pInfo->sOpts.nTxDescs*sizeof(TX_DESC)*i;
	}
	
	return TRUE;	
}

static void rhine_free_rings(PRHINE_INFO pInfo) {
	pci_free_consistent(pInfo->pcid,
	    pInfo->sOpts.nRxDescs * sizeof(RX_DESC) +
	    pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
   		pInfo->aRDRing, pInfo->pool_dma);
   	
	if (pInfo->tx_bufs)	   	
		pci_free_consistent(pInfo->pcid,
			    pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues,
		   		pInfo->tx_bufs, pInfo->tx_bufs_dma);
   		
}

static BOOL rhine_init_rd_ring(PRHINE_INFO pInfo) {
	int i;
	dma_addr_t		curr=pInfo->rd_pool_dma;
	PRX_DESC 		pDesc;
	
	/* Init the RD ring entries */
	for (i = 0; i < pInfo->sOpts.nRxDescs; i++, curr+=sizeof(RX_DESC)) {

		pDesc=&(pInfo->aRDRing[i]);
		
		pDesc->pInfo=alloc_rd_info();
		ASSERT(pDesc->pInfo);
	
		if (!rhine_alloc_rx_buf(pInfo, pDesc)) {
			RHINE_PRT(MSG_LEVEL_ERR,KERN_ERR "%s: can not alloc rx bufs\n",
			pInfo->dev->name);
			return FALSE;
		}
		
		pDesc->next=&(pInfo->aRDRing[(i+1) % pInfo->sOpts.nRxDescs]);
		pDesc->pInfo->curr_desc=cpu_to_le32(curr);
		pDesc->next_desc = cpu_to_le32(curr+sizeof(RX_DESC));	
	}

	pInfo->aRDRing[i-1].next_desc=cpu_to_le32(pInfo->rd_pool_dma);
	pInfo->pCurrRD=&(pInfo->aRDRing[0]);
	return TRUE;

}

static void rhine_free_rd_ring(PRHINE_INFO pInfo) {	
	int i;
	/* Init the RD ring entries */	
	for (i = 0; i < pInfo->sOpts.nRxDescs; i++) {
		PRX_DESC 		pDesc=&(pInfo->aRDRing[i]);
		PRHINE_RD_INFO	pRDInfo=pDesc->pInfo;
		if (pRDInfo->skb_dma) {
			pci_unmap_single(pInfo->pcid,pRDInfo->skb_dma,
    	       pInfo->rx_buf_sz, PCI_DMA_FROMDEVICE);
        }
        
        if (pRDInfo->skb) 
			dev_kfree_skb(pRDInfo->skb);
		
		kfree(pDesc->pInfo);
	}

}

static BOOL rhine_init_td_ring(PRHINE_INFO pInfo) {
	int i,j;
	dma_addr_t	curr;
	PTX_DESC 		pDesc;
	
	/* Init the RD ring entries */
	for (j=0;j<pInfo->nTxQueues;j++) {
		curr=pInfo->td_pool_dma[j];
		
		for (i = 0; i < pInfo->sOpts.nTxDescs; i++, curr+=sizeof(TX_DESC)) {
			pDesc=&(pInfo->apTDRings[j][i]);
			pDesc->pInfo=alloc_td_info();
			
			if (pDesc->pInfo==NULL) {
				RHINE_PRT(MSG_LEVEL_ERR,KERN_ERR "%s: can not alloc tx bufs\n",
					pInfo->dev->name);				
				return FALSE;
			}
												
			pDesc->pInfo->buf=pInfo->tx_bufs+(i+j)*PKT_BUF_SZ;
			pDesc->pInfo->buf_dma=pInfo->tx_bufs_dma+(i+j)*PKT_BUF_SZ;
	
			pDesc->next=&(pInfo->apTDRings[j][(i+1) % pInfo->sOpts.nTxDescs]);
			pDesc->pInfo->curr_desc=cpu_to_le32(curr);
			pDesc->tdesc1.f1Chain=1;
			pDesc->next_desc = cpu_to_le32(curr+sizeof(TX_DESC));
		}
		
		pInfo->apTDRings[j][i-1].next_desc=cpu_to_le32(pInfo->td_pool_dma[j]);
		pInfo->apTailTD[j]=pInfo->apCurrTD[j]=&(pInfo->apTDRings[j][0]);
		
	}	
	return TRUE;
}

static void rhine_free_td_ring(PRHINE_INFO pInfo) {
	int i, j;
	
	for (j=0;j<pInfo->nTxQueues;j++) {
		for (i = 0; i < pInfo->sOpts.nTxDescs; i++) {
			PTX_DESC 		pDesc=&(pInfo->apTDRings[j][i]);
			PRHINE_TD_INFO	pTDInfo=pDesc->pInfo;
	
			if (pTDInfo->skb_dma) 
				pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma,
        		   pTDInfo->skb->len, PCI_DMA_TODEVICE);
        	   
			if (pTDInfo->skb)
				dev_kfree_skb(pTDInfo->skb);
        	   
			kfree(pDesc->pInfo);
		}	
	}
	
}

/*-----------------------------------------------------------------*/
static int rhine_rx_srv(PRHINE_INFO pInfo, int status) {
	PRX_DESC				pRD=pInfo->pCurrRD;
	struct net_device_stats* pStats=&pInfo->stats;		 
	PMAC_REGS				pMacRegs=pInfo->pMacRegs;
	int						works=0;
    	
    for (pRD=pInfo->pCurrRD;
    	pRD->rdesc0.f1Owner==OWNED_BY_HOST;pRD=pRD->next) {
    	
	     //No more skb buff?
	    if (pRD->pInfo->skb==NULL) {    
			if (!rhine_alloc_rx_buf(pInfo,pRD))
	    		break; 
	    }	
	    	    	   		
    	if (works++>15)
    		break;
    		    	
		pInfo->adwRMONStats[RMON_Octets]+=pRD->rdesc0.f15Length;
		
		if (pRD->rdesc0.f15Length>=64) {
			if (pRD->rdesc0.f15Length==64)
				pInfo->adwRMONStats[RMON_Pkts64Octets]++;
			else if (pRD->rdesc0.f15Length<128)
				pInfo->adwRMONStats[RMON_Pkts65to127Octets]++;
			else if (pRD->rdesc0.f15Length<256)
				pInfo->adwRMONStats[RMON_Pkts128to255Octets]++;				
			else if (pRD->rdesc0.f15Length<512)
				pInfo->adwRMONStats[RMON_Pkts256to511Octets]++;
			else if (pRD->rdesc0.f15Length<1024)
				pInfo->adwRMONStats[RMON_Pkts512to1023Octets]++;
			else if (pRD->rdesc0.f15Length<=1518)
				pInfo->adwRMONStats[RMON_Pkts1024to1518Octets]++;			
		}		    		
		
    	if (pRD->rdesc0.byRSR1 & RSR1_RXOK) {
			if (rhine_receive_frame(pInfo, pRD)) {
				if (!rhine_alloc_rx_buf(pInfo,pRD)) {
					RHINE_PRT(MSG_LEVEL_ERR, KERN_ERR
					"%s: can not allocate rx buf\n", pInfo->dev->name);	
					pStats->rx_errors++;
					break;
				}				
			} else {
				pStats->rx_errors++;			
				pStats->rx_dropped++;
			}
		} else {
			pStats->rx_errors++;
			if (pRD->rdesc0.byRSR0 & RSR0_CRC)
				pStats->rx_crc_errors++;
			if (pRD->rdesc0.byRSR0 & RSR0_FAE)
				pStats->rx_frame_errors++;
			if (pRD->rdesc0.byRSR0 & RSR0_FOV)
				pStats->rx_fifo_errors++;
			pStats->rx_dropped++;
		}
		
		pRD->rdesc0.f1Owner=OWNED_BY_NIC;
		
		if (pInfo->flags & RHINE_FLAGS_FLOW_CTRL)
			writeb(1,&pMacRegs->byFlowCR0);
		pInfo->dev->last_rx=jiffies;
    }
    pInfo->pCurrRD=pRD;
    return works;
}

#ifdef VMNS
static void inline
rhine_attach_vmns_info(PRX_DESC pRD,struct sk_buff* skb,U16 wTag) {
	PVMNS_ATTACH_INFO pAttch = GET_VMNS_ATTACH_INFO(skb);
	
	
	pAttch->type = ATT_INFO_TYPE_NONE;
	
	if (pRD->rdesc1.byPQSTS & PQSTS_TAG) {
		pAttch->type |=ATT_INFO_TYPE_TAG;		
		if (pRD->rdesc0.byRSR1 & RSR1_VIDHIT)
			pAttch->type |=ATT_INFO_TYPE_VIDHIT;
		pAttch->vlan=wTag & 0xfff;
		pAttch->priority=(wTag>>12) & 0x7;	
	}
	
    if (pRD->rdesc1.byIPKT) {
       pAttch->type |= ATT_INFO_TYPE_INST;
	   pAttch->InstNo = pRD->rdesc1.byIPKT;
	}
	pAttch->protocol=skb->protocol;
	skb->protocol=VMNS_FRAME_TYPE;
}
#endif

static inline void 
rhine_rx_csum(PRX_DESC pRD, struct sk_buff* skb) {
	skb->ip_summed=CHECKSUM_NONE;	
	if (pRD->rdesc0.byRSR0 & RSR0_FRAG) 
		return;		
	if (pRD->rdesc1.byPQSTS & PQSTS_IPKT) {	
			if (pRD->rdesc1.byPQSTS & PQSTS_IPOK) {
			if ((pRD->rdesc1.byPQSTS & PQSTS_TCPKT) 
				||(pRD->rdesc1.byPQSTS & PQSTS_UDPKT)) {
				if (!(pRD->rdesc1.byPQSTS & PQSTS_TUOK)) {
					return;
				}
			}
			skb->ip_summed=CHECKSUM_UNNECESSARY;
		}
	}
}

static BOOL 
rhine_receive_frame(PRHINE_INFO pInfo, PRX_DESC pRD) {
   	PRHINE_RD_INFO pRDInfo=pRD->pInfo; 	
   	struct net_device_stats* pStats=&pInfo->stats;   	
	struct sk_buff* skb;	
	U16		wTag;
	
	if ((pRD->rdesc0.byRSR1 & (RSR1_STP|RSR1_EDP))
		!=(RSR1_STP|RSR1_EDP)) {
		RHINE_PRT(MSG_LEVEL_VERBOSE, 
			KERN_NOTICE " %s : the received frame span multple RDs\n",
			pInfo->dev->name);
		pStats->rx_length_errors++;
		return FALSE;
	}

	if ((pRD->rdesc1.byPQSTS & PQSTS_TAG) && 
		(!(pInfo->flags & 
			(RHINE_FLAGS_VMNS_COMMITTED|RHINE_FLAGS_TAGGING)
			))) {
		pStats->rx_dropped++;
		return FALSE;						
	}	
	//Drop small packet
	if (pRD->rdesc0.byRSR0 & RSR0_RUNT) {
		if (!(pRD->rdesc1.byPQSTS & PQSTS_TAG)) {		
			if (pRD->rdesc0.f15Length<60) {
				pStats->rx_length_errors++;
				pInfo->adwRMONStats[RMON_UndersizePkts]++;
				return FALSE;
			}
		} 
	}	
	
	//Drop long packet	
	if ((pRD->rdesc0.byRSR0 & RSR0_LONG) &&
		(!(pRD->rdesc1.byPQSTS & PQSTS_TAG)) &&
		(pRD->rdesc0.f15Length>1500)) {				
		pStats->rx_length_errors++;
		pInfo->adwRMONStats[RMON_OversizePkts]++;		
		return FALSE;			
	}

	if (pRD->rdesc0.byRSR1 & RSR1_MAR)
		pInfo->adwRMONStats[RMON_MulticastPkts]++;
		
	if (pRD->rdesc0.byRSR1 & RSR1_BAR)
		pInfo->adwRMONStats[RMON_BroadcastPkts]++;
								
	skb=pRDInfo->skb;		
	skb->dev=pInfo->dev;
	
	pci_unmap_single(pInfo->pcid,pRDInfo->skb_dma,
        		pInfo->rx_buf_sz, PCI_DMA_FROMDEVICE); 
        		
    pRDInfo->skb_dma=(dma_addr_t) NULL;
    
    if (pInfo->flags & RHINE_FLAGS_IP_ALIGN) {
    	int i;
    	for (i = pRD->rdesc0.f15Length+4; i >= 0 ; i--)
        	*(skb->data + i + 2) = *(skb->data + i);
		skb->data += 2;
		skb->tail += 2;		
    }
    
    //Get Tag
	wTag=htons(*(PU16)((PU8)skb->data+((pRD->rdesc0.f15Length+3) & ~3)+2));	
	
	skb_put(skb,(pRD->rdesc0.f15Length-4));

	skb->protocol=eth_type_trans(skb, skb->dev);
	
	//drop frame not met IEEE 802.3
	if (pInfo->flags & RHINE_FLAGS_VAL_PKT_LEN) {
		if ((skb->protocol==htons(ETH_P_802_2)) && 
			(skb->len!=ntohs(skb->mac.ethernet->h_proto))) {
			pStats->rx_length_errors++;			
			return FALSE;
		}
	}
	
	skb->ip_summed=CHECKSUM_NONE;	
	
	if (pInfo->flags & RHINE_FLAGS_RX_CSUM) 
		rhine_rx_csum(pRD,skb);

#ifdef VMNS
	if (pInfo->flags & RHINE_FLAGS_VMNS_COMMITTED)
		rhine_attach_vmns_info(pRD, skb, wTag);
#endif
	
	if (pRD->rdesc0.byRSR0 & RSR0_FRAG) 
		skb->ip_summed=CHECKSUM_NONE;		

	pStats->rx_bytes+=skb->len;
	pStats->rx_packets++;

	netif_rx(skb);
	
	return TRUE;
}

static BOOL rhine_alloc_rx_buf(PRHINE_INFO pInfo, PRX_DESC pRD) {
	PRHINE_RD_INFO pRDInfo=pRD->pInfo;
	
	pRDInfo->skb=dev_alloc_skb(pInfo->rx_buf_sz+15);

	if (pRDInfo->skb==NULL) {
		
		return FALSE;
	}	
	ASSERT(pRDInfo->skb);
	skb_reserve(pRDInfo->skb,((U32)pRDInfo->skb->tail) & 15);
	pRDInfo->skb->dev=pInfo->dev;
	pRDInfo->skb_dma=
		pci_map_single(pInfo->pcid, pRDInfo->skb->tail, pInfo->rx_buf_sz, 
			PCI_DMA_FROMDEVICE);
	
	*((PU32) &(pRD->rdesc0)) = 0;
	
	pRD->rdesc1.f15BufLen=cpu_to_le32(pInfo->rx_buf_sz);
	pRD->rdesc0.f1Owner=OWNED_BY_NIC;
	pRD->buff_addr=cpu_to_le32(pRDInfo->skb_dma);
	
	return TRUE;	
}

//
// Re-transmited the frame
//
static void rhine_tx_srv_resend(PRHINE_INFO pInfo, PTX_DESC pTD, int iQNo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;	
	
	WAIT_MAC_TX_OFF(pMacRegs);
		
	pTD->tdesc0.f1Owner=OWNED_BY_NIC;
	
	writel(cpu_to_le32(pTD->pInfo->curr_desc),
		&pMacRegs->adwCurrTxDescAddr[iQNo]);
		
	BYTE_REG_BITS_ON(CR0_TXON,&pMacRegs->byCR0);		
	
    if( pInfo->flags & RHINE_FLAGS_TAGGING)
	     BYTE_REG_BITS_ON(1 << (7-iQNo), &pMacRegs->byTXQWAK);
	     
	BYTE_REG_BITS_ON(CR1_TDMD1,&pMacRegs->byCR1);
	
}

//
//  Drop the frame
//
static void rhine_tx_srv_drop(PRHINE_INFO pInfo, PTX_DESC pTD, int iQNo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;	
	
	WAIT_MAC_TX_OFF(pMacRegs);
	

	pTD=pTD->next;
	
	writel(cpu_to_le32(pTD->pInfo->curr_desc),
		&pMacRegs->adwCurrTxDescAddr[iQNo]);	
		
	BYTE_REG_BITS_ON(CR0_TXON,&pMacRegs->byCR0);		
	
    if( pInfo->flags & RHINE_FLAGS_TAGGING)
	     BYTE_REG_BITS_ON(1 << (7-iQNo), &pMacRegs->byTXQWAK);
	     
	BYTE_REG_BITS_ON(CR1_TDMD1,&pMacRegs->byCR1);
	
}

//
// Drop all frame on the TX Queue
//
static void rhine_tx_srv_drop_all(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;	
	PTX_DESC	pTD;		
	int			iQNo;
	struct net_device_stats* pStats=&pInfo->stats;
	
	WAIT_MAC_TX_OFF(pMacRegs);		
	
	//Drop all transmited packets in the TD queue, because
	//The TD write back status may be incorrect.	

	for (iQNo=0;iQNo<pInfo->nTxQueues;iQNo++) {	
	    for (pTD=pInfo->apTailTD[iQNo];pInfo->iTDUsed[iQNo]>0;
	    	pTD=pTD->next) {	    	
	    	if (pTD->tdesc0.f1Owner==OWNED_BY_NIC) 
	    		break;	    	
			rhine_free_tx_buf(pInfo,pTD);	
		    pInfo->iTDUsed[iQNo]--;
		    pStats->tx_dropped++;
		}
		
	    pInfo->apTailTD[iQNo]=pTD;
	    writel(cpu_to_le32(pTD->pInfo->curr_desc),
	    	&(pMacRegs->adwCurrTxDescAddr[iQNo]));
	    	
	    if (pInfo->flags & RHINE_FLAGS_TAGGING) 
	    	BYTE_REG_BITS_ON(1<<(7-iQNo), &pMacRegs->byTXQWAK);	    
	    		    	
	}
	
	BYTE_REG_BITS_ON(CR0_TXON,&pMacRegs->byCR0);
	BYTE_REG_BITS_ON(CR1_TDMD1,&pMacRegs->byCR1);
}

static int rhine_tx_srv(PRHINE_INFO pInfo, int status) {
	PTX_DESC	pTD;	
	int			iQNo;
	BOOL		bFull=FALSE;
	int			works=0;

	struct net_device_stats* pStats=&pInfo->stats;
	
	for (iQNo=0;iQNo<pInfo->nTxQueues;iQNo++) {
	
	    for (pTD=pInfo->apTailTD[iQNo];
	    	pInfo->iTDUsed[iQNo]>0;
	    	pTD=pTD->next) {
	    				
	    	if (pTD->tdesc0.f1Owner==OWNED_BY_NIC)
	    		break;
	    
			if ((works++>15) && (!(status & (ISR_UDFI|ISR_ABTI))))
				break;    	
				
	    	//Only the status of first TD in the chain is correct
			if (pTD->tdesc1.byTCR & TCR_STP) { 
				if (pTD->tdesc0.byTSR1 & TSR1_TERR) {				
					RHINE_PRT(MSG_LEVEL_DEBUG, KERN_INFO
						"%s : td error %x,%x\n",pInfo->dev->name,
						pTD->tdesc0.byTSR1,pTD->tdesc0.byTSR0);
						
					pStats->tx_errors++;
					
					if (pTD->tdesc0.byTSR1 & TSR1_UDF) {				
						if (pInfo->sOpts.tx_thresh<4) {
							pInfo->sOpts.tx_thresh++;	
						
							RHINE_PRT(MSG_LEVEL_VERBOSE, KERN_INFO
									"%s: transmitter fifo underrun occurred, increase fifo threshold to  %d\n",
									pInfo->dev->name, pInfo->sOpts.tx_thresh);
								
							rhine_set_tx_thresh(pInfo,pInfo->sOpts.tx_thresh);
						}
						rhine_tx_srv_resend(pInfo,pTD,iQNo);
						pInfo->IntMask|=IMR_PTXM;
						pStats->tx_fifo_errors++;
						break;
					}				
				
					if (pTD->tdesc0.byTSR1 & TSR1_ABT) {	
						RHINE_PRT(MSG_LEVEL_VERBOSE, KERN_INFO
							"%s: transmitter fifo abort occurred\n",pInfo->dev->name);
						rhine_tx_srv_drop(pInfo,pTD,iQNo);
						pInfo->IntMask|=IMR_PTXM;						
						pStats->tx_aborted_errors++;
					}
				
					pStats->tx_dropped++;
					if (pTD->tdesc0.byTSR1 & TSR1_CRS) 
						pStats->tx_carrier_errors++;				
					if (pTD->tdesc0.byTSR1 & TSR1_OWC)
						pStats->tx_window_errors++;
					if (pTD->tdesc0.byTSR0 & TSR0_CDH)
						pStats->tx_heartbeat_errors++;				
				
				} else {
					pStats->collisions+=(pTD->tdesc0.byTSR0 & 0xF);
					pStats->tx_packets++;
					pStats->tx_bytes+=pTD->pInfo->skb->len;
				}
				rhine_free_tx_buf(pInfo,pTD);
			    pInfo->iTDUsed[iQNo]--;	
		    }
	    }	    
	    pInfo->apTailTD[iQNo]=pTD;
	    
	    if (AVAIL_TD(pInfo, iQNo) < 4 ) {
	    	bFull=TRUE;	
	    }
    }

	if (netif_queue_stopped(pInfo->dev) && (bFull==FALSE)
		&& (!(pInfo->mii_status & RHINE_LINK_FAIL))) {
			rhine_disable_tx_int(pInfo);
		netif_wake_queue(pInfo->dev);
	}
	return works;		
}

static void 
rhine_print_link_status(PRHINE_INFO pInfo) {
	
	if (pInfo->mii_status & RHINE_LINK_FAIL) {
		RHINE_PRT(MSG_LEVEL_INFO, KERN_NOTICE
			"%s: failed to detect cable link\n",pInfo->dev->name);
	}
	else {
		if (pInfo->mii_status & RHINE_AUTONEG_ENABLE)
			RHINE_PRT(MSG_LEVEL_INFO, KERN_NOTICE
				"%s: Link autonegation",pInfo->dev->name)			
		else
			RHINE_PRT(MSG_LEVEL_INFO, KERN_NOTICE
				"%s: Link forced",pInfo->dev->name);
				
		if (pInfo->mii_status & RHINE_SPEED_100) 
			RHINE_PRT(MSG_LEVEL_INFO," speed 100M bps")
		else 
			RHINE_PRT(MSG_LEVEL_INFO," speed 10M bps");
				
		if (pInfo->mii_status & RHINE_DUPLEX_FULL) 
			RHINE_PRT(MSG_LEVEL_INFO, " full duplex\n")
		else 
			RHINE_PRT(MSG_LEVEL_INFO, " half duplex\n");
			
	}
}

static void rhine_error(PRHINE_INFO pInfo, int status) {

	if (status & ISR_BE) {
		RHINE_PRT(MSG_LEVEL_ERR, KERN_ERR
			"%s: Hardware fatal error.\n", 
			pInfo->dev->name);
		netif_stop_queue(pInfo->dev);
		rhine_shutdown(pInfo);
		return;		
	}
	
	if (status & ISR_SRCI) {
		pInfo->mii_status=rhine_check_media_mode(pInfo);	
		rhine_print_link_status(pInfo);
	
		if (pInfo->byRevId>=REV_ID_VT6102_A)
			enable_flow_control_ability(pInfo);				
			
#ifdef VMNS	
		if (pInfo->flags & RHINE_FLAGS_VMNS_COMMITTED) {
	        PVMNS_DRV_PRIVATE pvmns_priv = pInfo->vmns_priv;
            if ((pInfo->flags & RHINE_FLAGS_VMNS_COMMITTED) && pvmns_priv->notify) {
            	if (pInfo->mii_status & RHINE_LINK_FAIL) 
	            	pvmns_priv->notify(pInfo->dev->name, VMNS_EVENT_LINK_FAIL);
				else 
					pvmns_priv->notify(pInfo->dev->name, VMNS_EVENT_LINK_UP);
			}
		}
#endif
		
		if (pInfo->mii_status & RHINE_LINK_FAIL) 
			netif_stop_queue(pInfo->dev);
		else 
			netif_wake_queue(pInfo->dev);

	};
	
	if (status & ISR_TDWBRAI) {
		RHINE_PRT(MSG_LEVEL_VERBOSE, KERN_INFO
			"%s: Tx descriptor status write back race occurred.\n", 
			pInfo->dev->name);
		rhine_tx_srv_drop_all(pInfo);
	}	

	if (status & ISR_CNT) {
		rhine_UpdateMIBCounter(pInfo);	
	}
	
#ifdef RHINE_DBG

	if (status & ISR_NORBF) {
		RHINE_PRT(MSG_LEVEL_VERBOSE, KERN_INFO
			"%s: No more receive buffer to be used\n",
			pInfo->dev->name);
	}

	if (status & ISR_OVFI) {
		RHINE_PRT(MSG_LEVEL_VERBOSE, KERN_INFO
			"%s: Received FIFO Overflow\n",
			pInfo->dev->name);	
	}		
	
#endif

}

static void rhine_free_tx_buf(PRHINE_INFO pInfo, PTX_DESC pDesc) {
	PRHINE_TD_INFO	pTDInfo=pDesc->pInfo;
	struct sk_buff* skb=pTDInfo->skb;

	if (pTDInfo->skb_dma && (pTDInfo->skb_dma != pTDInfo->buf_dma))	{
	
#ifdef RHINE_ZERO_COPY_SUPPORT
        pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma,
        	pDesc->tdesc1.f15BufLen,PCI_DMA_TODEVICE);
#else
		pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma,skb->len, 
			PCI_DMA_TODEVICE);
#endif			
	}
	
	dev_kfree_skb_irq(skb);
      
	pTDInfo->skb_dma=0;
	pTDInfo->skb=0;

}

static int  rhine_open(struct net_device *dev) {
	PRHINE_INFO	pInfo=(PRHINE_INFO) dev->priv;
	int i;
	
#ifdef VMNS	
	if (pInfo->flags & RHINE_FLAGS_OPENED)
		return -EBUSY;
#endif
	
	pInfo->rx_buf_sz=(dev->mtu <= 1504 ? PKT_BUF_SZ : dev->mtu + 32);

	if (!rhine_init_rings(pInfo)) {
		return -ENOMEM;
	}

	if (!rhine_init_rd_ring(pInfo)) {
		rhine_free_rings(pInfo);
		return -ENOMEM;
	}
	
	if (!rhine_init_td_ring(pInfo)) {
		rhine_free_rd_ring(pInfo);
		rhine_free_rings(pInfo);
		return -ENOMEM;		
	}
	
	rhine_init_registers(pInfo, RHINE_INIT_COLD);

	i=request_irq(pInfo->pcid->irq, &rhine_intr, SA_SHIRQ, dev->name, dev);
	
	if (i)
		return i;	
		
	rhine_disable_tx_int(pInfo);			
	rhine_enable_int(pInfo);

	netif_start_queue(dev);	
		
	pInfo->flags |=RHINE_FLAGS_OPENED;
		 	
	MOD_INC_USE_COUNT;
	return 0;
}

static void rhine_shutdown(PRHINE_INFO pInfo) {
	PMAC_REGS pMacRegs=pInfo->pMacRegs;		
	rhine_disable_int(pInfo);
	writeb(CR0_STOP,&pMacRegs->byCR0);
	SafeDisableMiiAutoPoll(pInfo);
	rhine_ClearISR(pInfo->pMacRegs);
}

static int  rhine_close(struct net_device *dev) {
	PRHINE_INFO	pInfo=(PRHINE_INFO) dev->priv;	

	netif_stop_queue(dev);

	rhine_shutdown(pInfo);

	if (pInfo->flags & RHINE_FLAGS_WOL_ENABLED)
		rhine_get_ip(pInfo);
			
	MOD_DEC_USE_COUNT;
	
	free_irq(dev->irq, dev);
	
	rhine_free_td_ring(pInfo);	
	rhine_free_rd_ring(pInfo);	
	rhine_free_rings(pInfo);
	
	pInfo->flags &=(~RHINE_FLAGS_OPENED);
	return 0;
}

static int  rhine_xmit(struct sk_buff *skb, struct net_device *dev) {
 	PRHINE_INFO	pInfo=dev->priv;
	int 			iQNo=0;
	PTX_DESC		pTD,pHeadTD;
	int				flags;
	PMAC_REGS		pMacRegs=pInfo->pMacRegs;
    
	spin_lock_irqsave(&pInfo->lock,flags);
    
	pTD=pHeadTD=pInfo->apCurrTD[0];
	
	pHeadTD->tdesc1.byTCR=(TCR_IC|TCR_EDP|TCR_STP);
	
#ifdef RHINE_ZERO_COPY_SUPPORT
	if (skb_shinfo(skb)->nr_frags>0) {
		int	nfrags=skb_shinfo(skb)->nr_frags;
		pTD->pInfo->skb=skb;
		
		if ((AVAIL_TD(pInfo,iQNo)<nfrags) ||
			(pInfo->flags & RHINE_FLAGS_TX_ALIGN)) {
			skb_linearize(skb,GFP_ATOMIC);
			memcpy(pTD->pInfo->buf,skb->data,skb->len);
			pTD->pInfo->skb_dma=pTD->pInfo->buf_dma;
    	   	pTD->buff_addr=cpu_to_le32(pTD->pInfo->skb_dma);
       		pTD->tdesc1.f15BufLen=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);			
		}		
		else {        	        	
			int i;
	        pTD->pInfo->skb_dma = 
	        	pci_map_single(pInfo->pcid, skb->data, 
    	    		skb->len -skb->data_len, PCI_DMA_TODEVICE);

	       	pTD->buff_addr=cpu_to_le32(pTD->pInfo->skb_dma);
	       	pTD->tdesc1.f15BufLen=skb->len - skb->data_len;
	       	
			for (i=0;i<nfrags;i++) {
				skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
				void* addr = ((void *) page_address(frag->page + 
							frag->page_offset));
							
				pTD=pTD->next;
				pTD->pInfo->skb_dma=
					pci_map_single(pInfo->pcid,addr,frag->size,PCI_DMA_TODEVICE);
		       	pTD->buff_addr=cpu_to_le32(pTD->pInfo->skb_dma);
		       	pTD->tdesc1.f15BufLen=frag->size;
			}			
			pTD->tdesc1.byTCR|=TCR_EDP;
		} 
		
	}
	else
#endif		
	if 	(((pInfo->flags & RHINE_FLAGS_TX_ALIGN)  
    		&& ((long)skb->data & 3))||
    		(skb->len<ETH_ZLEN)) {    	
    	    	
		memcpy(pHeadTD->pInfo->buf,skb->data,skb->len);
	    pHeadTD->pInfo->skb=skb;		
		pHeadTD->pInfo->skb_dma=pHeadTD->pInfo->buf_dma;
       	pHeadTD->buff_addr=cpu_to_le32(pHeadTD->pInfo->skb_dma);
       	pHeadTD->tdesc1.f15BufLen=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);
       //padding zero as packet size is less than 60 bytes
       if (skb->len<ETH_ZLEN)
       		memset(pHeadTD->pInfo->buf+skb->len,
       			0,
       			ETH_ZLEN-skb->len);
    } 
    else {     	
	    pHeadTD->pInfo->skb=skb;    
        pHeadTD->pInfo->skb_dma = pci_map_single(pInfo->pcid, skb->data, skb->len, 
        	PCI_DMA_TODEVICE);
       	pHeadTD->buff_addr=cpu_to_le32(pHeadTD->pInfo->skb_dma);
       	pHeadTD->tdesc1.f15BufLen=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);
    }
            
#ifdef VMNS    
    if (pInfo->flags & RHINE_FLAGS_VMNS_COMMITTED) {
    	PVMNS_ATTACH_INFO pAttch = GET_VMNS_ATTACH_INFO(skb);
        if (pAttch->type & ATT_INFO_TYPE_TAG) {
        	pHeadTD->tdesc0.f12VID=(pAttch->vlan & 0xfff);
        	pHeadTD->tdesc0.f3Priority=(pAttch->priority & 0x7);
        	pHeadTD->tdesc1.byTCR|=TCR_TAG;
        }                                                        
    }
    else
#endif
	if (pInfo->flags & RHINE_FLAGS_TAGGING) {
       	pHeadTD->tdesc0.f12VID=(pInfo->sOpts.vid & 0xfff);
       	pHeadTD->tdesc0.f3Priority=0;
       	pHeadTD->tdesc1.byTCR|=TCR_TAG;
	} 

	
#ifdef RHINE_TX_CSUM_SUPPORT
	if ((pInfo->flags & RHINE_FLAGS_TX_CSUM) &&
		(skb->ip_summed==CHECKSUM_HW)) {
		struct iphdr* ip=skb->nh.iph;		
		if (ip->protocol==IPPROTO_TCP) 
			pHeadTD->tdesc1.byTCR|=TCR_TCPCK;
		else if (ip->protocol==IPPROTO_UDP)
			pHeadTD->tdesc1.byTCR|=(TCR_UDPCK);
		pHeadTD->tdesc1.byTCR|=TCR_IPCK;			
	}	
#endif

	wmb();
	pHeadTD->tdesc0.f1Owner=OWNED_BY_NIC;
	wmb();
	
	pInfo->iTDUsed[iQNo]++;
	
	if (AVAIL_TD(pInfo,iQNo)<=1) {
	    netif_stop_queue(dev);
	    rhine_enable_tx_int(pInfo);
    }
    
	pInfo->apCurrTD[iQNo]=pTD->next;  
	       		
    if (pInfo->flags & RHINE_FLAGS_TAGGING) 
	    BYTE_REG_BITS_ON(1 << (7-iQNo), &pMacRegs->byTXQWAK);
	    	    
    BYTE_REG_BITS_ON(CR1_TDMD1,&(pMacRegs->byCR1));
	
    dev->trans_start = jiffies;
    
	spin_unlock_irqrestore(&pInfo->lock,flags);
	return 0;
}

static void rhine_intr(int irq, void *dev_instance, struct pt_regs *regs) {
	struct net_device* dev=dev_instance;
	PRHINE_INFO	pInfo=(PRHINE_INFO) dev->priv;
	U32				isr_status;
	PMAC_REGS		pMacRegs=pInfo->pMacRegs;
	int				max_count=0;
	int				flags;

	isr_status=rhine_ReadISR(pMacRegs,pInfo->byRevId);
	
	if (isr_status==0) 
		return;	

	rhine_disable_int(pInfo);		
	spin_lock_irqsave(&pInfo->lock,flags);
				
	while (isr_status!=0) {		
			
		rhine_WriteISR(isr_status,pMacRegs,pInfo->byRevId);			
		
		if (isr_status & (ISR_SRCI|ISR_TDWBRAI|ISR_BE|ISR_CNT
#ifdef RHINE_DBG	
			|ISR_OVFI|ISR_UDFI))			
#else
			))
#endif			
			rhine_error(pInfo, isr_status);
			
		if (isr_status & (ISR_RXE|ISR_PRX)) 
			max_count+=rhine_rx_srv(pInfo, isr_status);

		if (isr_status & (ISR_TXE|ISR_PTX))
			max_count+=rhine_tx_srv(pInfo, isr_status);
				
		isr_status=rhine_ReadISR(pMacRegs,pInfo->byRevId);		
		
		if (max_count>pInfo->sOpts.int_works)
				break;
	}	
	spin_unlock_irqrestore(&pInfo->lock,flags);
	rhine_enable_int(pInfo);	

}

void rhine_get_cam_mask(PRHINE_INFO pInfo, PU32 pMask, 
	RHINE_CAM_TYPE cam_type) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	
    // enable CAMEN
    if (cam_type==RHINE_VLAN_ID_CAM)
	    writeb(CAMC_CAMEN | CAMC_VCAMSL, &pMacRegs->byCAMCR);
	else
		writeb(CAMC_CAMEN,&pMacRegs->byCAMCR);
	wmb();
    // read mask       
    *pMask = readl(&pMacRegs->dwCAMMASK);
    
    // disable CAMEN
    writeb(0, &pMacRegs->byCAMCR);
    
}

void rhine_set_cam_mask(PRHINE_INFO pInfo, U32 mask, 
	RHINE_CAM_TYPE cam_type) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
    if (cam_type==RHINE_VLAN_ID_CAM)
	    writeb(CAMC_CAMEN | CAMC_VCAMSL, &pMacRegs->byCAMCR);
	else
		writeb(CAMC_CAMEN,&pMacRegs->byCAMCR);
	wmb();
    // write mask       
    writel(mask, &pMacRegs->dwCAMMASK);
	    
    // disable CAMEN
    writeb(0, &pMacRegs->byCAMCR);
	
}

void rhine_set_cam(PRHINE_INFO pInfo, int idx, PU8 addr,
	RHINE_CAM_TYPE cam_type) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	int i;
	
    if (cam_type==RHINE_VLAN_ID_CAM)
	    writeb(CAMC_CAMEN | CAMC_VCAMSL, &pMacRegs->byCAMCR);
	else
		writeb(CAMC_CAMEN,&pMacRegs->byCAMCR);
		
	wmb();	
	writeb((U8)(idx & 0x1F), &pMacRegs->byCAMADD);	
	
    if (cam_type==RHINE_VLAN_ID_CAM) {
	    writeb(*addr, &pMacRegs->abyMAR[6]);
	    writeb(*(addr+1), &pMacRegs->abyMAR[7]);	    
	}
	else
		for (i=0;i<6;i++, addr++) 
			writeb(*addr,&(pMacRegs->abyMAR[i]));
	udelay(10);			
	wmb();			
	writeb(CAMC_CAMWR|CAMC_CAMEN,&pMacRegs->byCAMCR);				
	udelay(10);	
	
	writeb(0, &pMacRegs->byCAMCR);		
}

void rhine_get_cam(PRHINE_INFO pInfo, int idx, PU8 addr,
	RHINE_CAM_TYPE cam_type) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	int i;
	
    if (cam_type==RHINE_VLAN_ID_CAM)
	    writeb(CAMC_CAMEN | CAMC_VCAMSL, &pMacRegs->byCAMCR);
	else
		writeb(CAMC_CAMEN,&pMacRegs->byCAMCR);
		
	wmb();	
	writeb((U8)(idx & 0x1F), &pMacRegs->byCAMADD);	
	wmb();
	
	writeb(CAMC_CAMRD|CAMC_CAMEN,&pMacRegs->byCAMCR);
	
	wmb();
	udelay(10);
	
    if (cam_type==RHINE_VLAN_ID_CAM)
	    *((PU16) addr)=readw(&(pMacRegs->abyMAR[6]));
	else
		for (i=0;i<6;i++, addr++) 
			*((PU8)addr)=readb(&(pMacRegs->abyMAR[i]));		
		
	writeb(0, &pMacRegs->byCAMCR);		
}

static unsigned const ethernet_polynomial = 0x04c11db7U;
static inline u32 ether_crc(int length, unsigned char *data)
{
	int crc = -1;
    
	while(--length >= 0) {
		unsigned char current_octet = *data++;
		int bit;
		for (bit = 0; bit < 8; bit++, current_octet >>= 1) {
			crc = (crc << 1) ^
				((crc < 0) ^ (current_octet & 1) ? ethernet_polynomial : 0);
    	}
	}
	return crc;
}
                                                                            
static void rhine_set_multi(struct net_device *dev) {
	PRHINE_INFO			pInfo		=	(PRHINE_INFO) dev->priv;
	PMAC_REGS			pMacRegs	=	pInfo->pMacRegs;
	u32 				mc_filter[2];
	u8 					rx_mode;
	int					i;
	int					flags;			
	struct dev_mc_list	*mclist;
	
	spin_lock_irqsave(&pInfo->lock,flags); 	
	if (dev->flags & IFF_PROMISC) {			/* Set promiscuous. */
		/* Unconditionally log net taps. */
		printk(KERN_NOTICE "%s: Promiscuous mode enabled.\n", dev->name);
		writel(0xffffffff, &pMacRegs->abyMAR[0]);
		writel(0xffffffff, &pMacRegs->abyMAR[4]);			
		rx_mode = (RCR_AM|RCR_AB|RCR_PROM);	
	} 
	else if ((dev->mc_count > pInfo->multicast_limit)
		||  (dev->flags & IFF_ALLMULTI)) {
		writel(0xffffffff, &pMacRegs->abyMAR[0]);
		writel(0xffffffff, &pMacRegs->abyMAR[4]);
		rx_mode = (RCR_AM|RCR_AB);
	} 
	else if (pInfo->flags & RHINE_FLAGS_HAVE_CAM) {
		U32	mask=0;
		int offset=MCAM_SIZE-pInfo->multicast_limit;
		rhine_get_cam_mask(pInfo,&mask,RHINE_MULTICAST_CAM);
		for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
			i++, mclist = mclist->next) {			
		 	rhine_set_cam(pInfo,i+offset,mclist->dmi_addr,RHINE_MULTICAST_CAM);
	 		mask|=1<<(offset+i);
		}
		rhine_set_cam_mask(pInfo,mask,RHINE_MULTICAST_CAM);
		rx_mode=(RCR_AM|RCR_AB);
	}
	else {
		memset(mc_filter, 0, sizeof(mc_filter));
		for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
			 i++, mclist = mclist->next) {
			int bit_nr = ether_crc(ETH_ALEN, mclist->dmi_addr) >> 26;
			mc_filter[bit_nr >> 5] |= cpu_to_le32(1 << (bit_nr & 31));
		}
		writel(mc_filter[0], &pMacRegs->abyMAR[0]);
		writel(mc_filter[1], &pMacRegs->abyMAR[4]);
		rx_mode = (RCR_AM|RCR_AB);
	}
	
	writeb((pInfo->sOpts.rx_thresh<<5) | rx_mode, &pMacRegs->byRCR);
	spin_unlock_irqrestore(&pInfo->lock,flags);
}

static struct net_device_stats *rhine_get_stats(struct net_device *dev) {
	PRHINE_INFO	pInfo=(PRHINE_INFO) dev->priv;	
	return &pInfo->stats;
}


static int	rhine_ioctl(struct net_device *dev, struct ifreq *rq, int cmd) {

    switch(cmd) {    
#ifdef VMNS
    case VMNS_DRV_SIOC: {
		PVMNS_DRV_SIOC_HEADER   pParams = (PVMNS_DRV_SIOC_HEADER) rq->ifr_data;
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;
		if (vmns_process_ioctl(dev, pParams) == 0)
			return 0;
		}
		return -EAGAIN;                                                            	
#endif

#ifdef RHINE_ETHTOOL_IOCTL_SUPPORT
	case SIOCETHTOOL:		
		return rhine_ethtool_ioctl(dev, rq); 
		break;
#endif

#ifdef RHINE_MII_IOCTL_SUPPORT
	case SIOCGMIIPHY:		/* Get address of MII PHY in use. */
	case SIOCGMIIREG:		/* Read MII PHY register. */
	case SIOCSMIIREG:		/* Write to MII PHY register. */
		return rhine_mii_ioctl(dev, rq, cmd);
		break;
#endif	

    default:
        return -EOPNOTSUPP;
    }
	return 0;
}



/*------------------------------------------------------------------*/

MODULE_DEVICE_TABLE(pci, rhine_id_table);

static struct pci_driver rhine_driver = {
        name:		RHINE_NAME,
        id_table:	rhine_id_table,
        probe:		rhine_found1,
        remove:		rhine_remove1,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM			
		suspend:	rhine_suspend,
		resume:		rhine_resume,
#endif
#endif
}; 

static int __init rhine_init_module(void)
{
	int ret;
	ret=pci_module_init(&rhine_driver);
	
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM	
	register_inetaddr_notifier(&rhine_inetaddr_notifier);
	if(ret >= 0)
		register_reboot_notifier(&rhine_notifier);
	
#endif	
#endif

    return ret;
}

static void __exit rhine_cleanup_module(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM	
	unregister_reboot_notifier(&rhine_notifier);
	unregister_inetaddr_notifier(&rhine_inetaddr_notifier);
#endif	
#endif
    pci_unregister_driver(&rhine_driver);
}
 
module_init(rhine_init_module);
module_exit(rhine_cleanup_module);

/************************************************************************
* MII access , media link mode setting functions
************************************************************************/

static void SafeDisableMiiAutoPoll (PRHINE_INFO pInfo)
{
    WORD    	ww;
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;

    // turn off MAUTO
    writeb(0,&pMacRegs->byMIICR);

    // for VT86C100A only
    if (pInfo->byRevId <= REV_ID_VT86C100A_E) {
        // turn off MSRCEN
        // NOTE.... address of MII should be 0x01,
        //          otherwise SRCI will invoked
        writeb(1,&pMacRegs->byMIIAD);
        
        mdelay(1);

        // turn on MAUTO
        writeb(MIICR_MAUTO,&pMacRegs->byMIICR);
        
        // W_MAX_TIMEOUT is the timeout period
        for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
        	udelay(5);
        	if (readb(&pMacRegs->byMIIAD) & MIIAD_MDONE)
        		break;
        }
        
        // as soon as MDONE is on, 
        // this is the right time to turn off MAUTO
        writeb(0,&pMacRegs->byMIICR);
    }
    else {
        // as soon as MIDLE is on, MAUTO is really stoped
        for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
        	udelay(5);
        	if (BYTE_REG_BITS_IS_ON(MIIAD_MIDLE, &pMacRegs->byMIIAD))
        		break;
        }
    }
}

static void  EnableMiiAutoPoll(PMAC_REGS pMacRegs) {
	int ii;	
	writeb(0,&(pMacRegs->byMIICR));
	writeb(MIIAD_MSRCEN|0x01,&pMacRegs->byMIIAD);	
	writeb(MIICR_MAUTO,&pMacRegs->byMIICR);	
	
	for (ii=0;ii<W_MAX_TIMEOUT; ii++)
		if (BYTE_REG_BITS_IS_ON(MIIAD_MDONE, &pMacRegs->byMIIAD))
			break;
	BYTE_REG_BITS_ON(MIIAD_MSRCEN,&pMacRegs->byMIIAD);
}
static BOOL rhine_mii_read(PRHINE_INFO pInfo, U8 byIdx, PU16 pdata)
{
    WORD    	ww;
//    BYTE    	byOrgMIIAD;
//    BYTE    	byOrgMIICR;
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
//	byOrgMIIAD=readb(&pMacRegs->byMIIAD);
//	byOrgMIICR=readb(&pMacRegs->byMIICR);
		
    // disable MIICR_MAUTO, so that mii addr can be set normally
    SafeDisableMiiAutoPoll(pInfo);
    

	writeb(byIdx,&pMacRegs->byMIIAD);
	
	BYTE_REG_BITS_ON(MIICR_RCMD, &pMacRegs->byMIICR);	
		
    for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
    	if (!(readb(&pMacRegs->byMIICR) & MIICR_RCMD)) 
    		break;
    }
    
	*pdata=readw(&pMacRegs->wMIIDATA);
	
    // for VT3043 only
    if (pInfo->chip_id==VT86C100A)
		mdelay(1);
		
    // restore MIICR
//    writeb(byOrgMIIAD, &pMacRegs->byMIIAD);
//    writeb(byOrgMIICR, &pMacRegs->byMIICR);
    
    EnableMiiAutoPoll(pMacRegs);    
    if (ww == W_MAX_TIMEOUT) {
        return FALSE;
    }
    return TRUE;
}

static BOOL 
rhine_mii_write (PRHINE_INFO pInfo, BYTE byMiiAddr, WORD wData)
{
    WORD    	ww;
//    BYTE    	byOrgMIIAD;
//    BYTE    	byOrgMIICR;
   	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
   	

    // keep original MIICR value
//	byOrgMIIAD=readb(&pMacRegs->byMIIAD);
//	byOrgMIICR=readb(&pMacRegs->byMIICR);

    // disable MIICR_MAUTO, so that mii addr can be set normally
    SafeDisableMiiAutoPoll(pInfo);

    // MII reg offset
    writeb(byMiiAddr, &pMacRegs->byMIIAD);
    // set MII data
    writew(wData,&pMacRegs->wMIIDATA);

    // turn on MIICR_WCMD
    BYTE_REG_BITS_ON(MIICR_WCMD, &pMacRegs->byMIICR);
    
    // W_MAX_TIMEOUT is the timeout period
    for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
    	udelay(5);
    	if (!(readb(&pMacRegs->byMIICR) & MIICR_WCMD))
            break;
    }
    
    // for VT3043 only
    if (pInfo->chip_id==VT86C100A)
		mdelay(1);
		
    // restore MIICR
//    writeb(byOrgMIIAD, &pMacRegs->byMIIAD);
//    writeb(byOrgMIICR, &pMacRegs->byMIICR);
    
    EnableMiiAutoPoll(pMacRegs);    
    
    if (ww == W_MAX_TIMEOUT) {
        return FALSE;
    }
    
    return TRUE;
}

//
// Get the media mode stored in EEPROM or module options
//
static U32 rhine_get_opt_media_mode(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	U8			byRevId=pInfo->byRevId;
	U32			status=0;
	
	if (byRevId<REV_ID_VT6105_A0) {
		if (BYTE_REG_BITS_IS_ON(CFGC_MEDEN,&pMacRegs->byCFGC)) {
			if (!BYTE_REG_BITS_IS_ON(BCR0_MED2,&pMacRegs->byBCR0)) {			
				if (BYTE_REG_BITS_IS_ON(BCR1_MED1,&pMacRegs->byBCR1)) {
					if (BYTE_REG_BITS_IS_ON(BCR1_MED0,&pMacRegs->byBCR1))
						status=RHINE_SPEED_100|RHINE_DUPLEX_FULL;	
					else
						status=RHINE_SPEED_10|RHINE_DUPLEX_FULL;		
				} else {
					if (BYTE_REG_BITS_IS_ON(BCR1_MED0,&pMacRegs->byBCR1))
						status=RHINE_SPEED_100;		
					else
						status=RHINE_SPEED_10;
				}
				pInfo->mii_status=status;
				return status;
			} 
			status=RHINE_AUTONEG_ENABLE;
			pInfo->mii_status=status;
			return status;
		}
	} 		
	switch (pInfo->sOpts.spd_dpx) {
	case SPD_DPX_AUTO:
		status=RHINE_AUTONEG_ENABLE;
		break;
	case SPD_DPX_100_FULL:
		status=RHINE_SPEED_100|RHINE_DUPLEX_FULL;
		break;
	case SPD_DPX_10_FULL:
		status=RHINE_SPEED_10|RHINE_DUPLEX_FULL;
		break;
	case SPD_DPX_100_HALF:
		status=RHINE_SPEED_100;
		break;		
	case SPD_DPX_10_HALF:
		status=RHINE_SPEED_10;
		break;
	}
	pInfo->mii_status=status;
	return status;
}

static void mii_get_phy_id(PRHINE_INFO pInfo) {
	if (pInfo==NULL)
		return;
	rhine_mii_read(pInfo, MII_REG_PHYID2, (PU16) &pInfo->dwPHYId);
	rhine_mii_read(pInfo, MII_REG_PHYID1,	((PU16) &pInfo->dwPHYId) + 1);
}

static void mii_set_auto_on(PRHINE_INFO pInfo)
{                                                               	
	U8			byRevId=pInfo->byRevId;								
		
	if (byRevId>=REV_ID_VT6102_A) {	
		if ((byRevId>=REV_ID_VT6105_A0) || IS_PHY_VT6103(pInfo))
    		MII_REG_BITS_OFF(1, MII_REG_MODCFG, pInfo);	
    		    		
        if (MII_REG_BITS_IS_ON(BMCR_AUTO, MII_REG_BMCR ,pInfo))
            MII_REG_BITS_ON(BMCR_REAUTO, MII_REG_BMCR ,pInfo);
        else 														
			MII_REG_BITS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo);		
    } else 														
    	MII_REG_BITS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo);
}


static void mii_set_auto_off(PRHINE_INFO pInfo)
{                                                               
	U8	byRevId=pInfo->byRevId;
    
    if ((byRevId>=REV_ID_VT6105_A0) || 
    	((byRevId>=REV_ID_VT6102_A) && IS_PHY_VT6103(pInfo)))
    	MII_REG_BITS_ON(1, MII_REG_MODCFG, pInfo);
    	
    MII_REG_BITS_OFF(BMCR_AUTO, MII_REG_BMCR, pInfo);
}

//
// This function check if MAC operation at Full duplex mode
//
static BOOL rhine_is_full_duplex (PRHINE_INFO pInfo)
{
	U8	byRevId=pInfo->byRevId;
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
    // if in AUTO-NEGO mode
    if (byRevId>=REV_ID_VT6105_A0)
    	return BYTE_REG_BITS_IS_ON(MIISR_N_FDX, &pMacRegs->byMIISR);

    if (MII_REG_BITS_IS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo)) {

        // if my TX_FD on, check both TX_FD
        if (MII_REG_BITS_IS_ON(ANAR_TXFD, MII_REG_ANAR, pInfo)) {
            // partner's TX_FD
            if (MII_REG_BITS_IS_ON(ANLPAR_TXFD, MII_REG_ANLPAR, pInfo))
                return TRUE;
        }

        // if my T4 on, check both T4
        if (MII_REG_BITS_IS_ON(ANAR_T4, MII_REG_ANAR, pInfo)) {
            // partner's T4
            if (MII_REG_BITS_IS_ON(ANLPAR_T4, MII_REG_ANLPAR, pInfo))
                return FALSE;
        }

        // if my TX_HD on, check both TX_HD
        if (MII_REG_BITS_IS_ON(ANAR_TX, MII_REG_ANAR, pInfo)) {
            // partner's TX_HD
            if (MII_REG_BITS_IS_ON(ANLPAR_TX, MII_REG_ANLPAR, pInfo))
                return FALSE;
        }

        // if my 10_FD on, check both 10_FD
        if (MII_REG_BITS_IS_ON(ANAR_10FD, MII_REG_ANAR, pInfo)) {
            // partner's 10_FD
            if (MII_REG_BITS_IS_ON(ANLPAR_10FD, MII_REG_ANLPAR, pInfo))
                return TRUE;
        }

        // if all above is not, then it would be 10_HD or no link
        // both case will be half duplex
        return FALSE;
    }
    // if in force mode
    else {
    
        if (!MII_REG_BITS_IS_ON(BMCR_FDX, MII_REG_BMCR, pInfo))
            return FALSE;
    }

    return TRUE;
}

int rhine_set_media_mode(PRHINE_INFO pInfo, U32 mii_status) {
	U8			byRevId=pInfo->byRevId;
	U32			curr_status;	
	
	pInfo->mii_status=rhine_check_media_mode(pInfo);

	curr_status= pInfo->mii_status & (~RHINE_LINK_FAIL);


	if (byRevId>=REV_ID_VT6102_A) {
		//Enable or Disable PAUSE in ANAR	
		if (pInfo->sOpts.flow_cntl==2) 
			MII_REG_BITS_OFF(ANAR_PAUSE, MII_REG_ANAR, pInfo); 		
		else if (pInfo->sOpts.flow_cntl==3)
			MII_REG_BITS_ON(ANAR_PAUSE, MII_REG_ANAR, pInfo);	
	}
			
	//Check if new status is consisent with current status
    if (((mii_status & curr_status) & RHINE_AUTONEG_ENABLE) 
    	|| (mii_status==curr_status)) /*&& 
    	(!(pInfo->mii_status & RHINE_LINK_FAIL)))*/ {
		pInfo->mii_status=rhine_check_media_mode(pInfo);  
    	return 0;
    }
    
	// if connection type is AUTO    
    if (mii_status & RHINE_AUTONEG_ENABLE) {
        // set duplex mode of MAC according to duplex mode of MII
        if (rhine_is_full_duplex(pInfo))
            rhine_set_duplex(pInfo, TRUE);
        else
            rhine_set_duplex(pInfo,  FALSE);
            
		MII_REG_BITS_ON(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10, 
						MII_REG_ANAR, pInfo);

		// enable AUTO-NEGO mode
		mii_set_auto_on(pInfo);
    }
    // if not, then
    // 1). turn off AUTO-NEGO
    // 2). set USER-FORCED speed (10/100) & duplex (half/full) mode of MII and MAC
    else if (byRevId<REV_ID_VT6102_A) {
		U16    wBMCR;    
        // turn off AUTO-NEGO
                
		mii_set_auto_off(pInfo);
			
        // set loopback in MII to un-link in 100M mode,
        // in 10M mode set this bit cannot make it un-link
        // but it doesn't matter
        MII_REG_BITS_ON(BMCR_LBK, MII_REG_BMCR, pInfo);

        // read the original value of BMCR register
        rhine_mii_read(pInfo,MII_REG_BMCR, &wBMCR);
        
        // mask off AUTO-NEGO
        wBMCR &= ~(BMCR_AUTO);
        
        if (mii_status & RHINE_DUPLEX_FULL) {
        	wBMCR |=BMCR_FDX;
        	rhine_set_duplex(pInfo, TRUE);
        }
		else {
			wBMCR &=(~BMCR_FDX);
			rhine_set_duplex(pInfo, FALSE);
		}
	
		if (mii_status & RHINE_SPEED_100) 
			wBMCR |=BMCR_SPEED;
		else
			wBMCR &=(~BMCR_SPEED);
        	
        // write the setting to BMCR register
        rhine_mii_write(pInfo,	MII_REG_BMCR, wBMCR);

        // delay to avoid link down from force-10M to force-100M (only for 3065 in Win95/Win98)
        if (byRevId >= REV_ID_VT6102_A)
			mdelay(200);
			
        // unset MII loopback to re-link
        MII_REG_BITS_OFF(BMCR_LBK, MII_REG_BMCR, pInfo);
		
    }
    //For VT6105 & VT6102
    else {
    	WORD	wANAR;
    	
		mii_set_auto_off(pInfo);
		rhine_mii_read(pInfo, MII_REG_ANAR, &wANAR);
		
		wANAR &= (~(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10));
	
        if (mii_status & RHINE_SPEED_100) {
        	if (mii_status & RHINE_DUPLEX_FULL) {
	        	wANAR |=ANAR_TXFD;
	        	rhine_set_duplex(pInfo, TRUE);	
			}
			else {
				wANAR |=ANAR_TX;
	        	rhine_set_duplex(pInfo, FALSE);		
			}
        }
		else {
        	if (mii_status & RHINE_DUPLEX_FULL) {
	        	wANAR |=ANAR_10FD;
	        	rhine_set_duplex(pInfo, TRUE);	
			}
			else {
				wANAR |=ANAR_10;
	        	rhine_set_duplex(pInfo, FALSE);		
			}	
		}
	
		rhine_mii_write(pInfo,MII_REG_ANAR, wANAR);
		
        if (MII_REG_BITS_IS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo))
            MII_REG_BITS_ON(BMCR_REAUTO, MII_REG_BMCR, pInfo);
        else
            MII_REG_BITS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo);
    }    
    pInfo->mii_status=rhine_check_media_mode(pInfo);
	return RHINE_LINK_CHANGE;
}

U32 rhine_check_media_mode(PRHINE_INFO pInfo) {
	PMAC_REGS			pMacRegs=pInfo->pMacRegs;
	U32					status=0;
	
	if (BYTE_REG_BITS_IS_ON(MIISR_LNKFL, &pMacRegs->byMIISR)) {
		U16	w;
		status|=RHINE_LINK_FAIL;				
		rhine_mii_read(pInfo, MII_REG_ANAR, &w);
		if (w & ANAR_TX) {
			status|=RHINE_SPEED_100;
			if (w & ANAR_TXFD) {
				status|=RHINE_DUPLEX_FULL;
				rhine_set_duplex(pInfo,TRUE);
			}
			else 
				rhine_set_duplex(pInfo,FALSE);			
		}
		else {
			status|=RHINE_SPEED_10;
			if (w & ANAR_10FD) {
				status|=RHINE_DUPLEX_FULL;
				rhine_set_duplex(pInfo,TRUE);
			}
			else 
				rhine_set_duplex(pInfo,FALSE);			
		}
		
	}		
	else {	
	    if (BYTE_REG_BITS_IS_ON(MIISR_SPEED, &pMacRegs->byMIISR))
    	    status|=RHINE_SPEED_10;
	    else 
    	    status|=RHINE_SPEED_100;
    	    
	    /* check duplex mode*/    
	    /* if VT6105, check N_FDX bit in MII Status Register directly */
	    if (rhine_is_full_duplex(pInfo)) {
    		rhine_set_duplex(pInfo,TRUE);
	    	status|=RHINE_DUPLEX_FULL;
	    }	
	    else {
		   	rhine_set_duplex(pInfo,FALSE);
		}    	    
    }
	   	
	if (MII_REG_BITS_IS_ON(BMCR_AUTO,MII_REG_BMCR,pInfo)) {
		U16	w;
		rhine_mii_read(pInfo, MII_REG_ANAR, &w);
		if ((w & (ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10))
			==(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10)) {
			status|=RHINE_AUTONEG_ENABLE;
		}
	}    
    return status;
}

static void enable_flow_control_ability(PRHINE_INFO pInfo) {

	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	U16			wANAR,wANLPAR;
	U8			byFlowCR;

	if (pInfo->byRevId<REV_ID_VT6102_A)
		return;
	
	rhine_mii_read(pInfo,MII_REG_ANAR,&wANAR);
	rhine_mii_read(pInfo,MII_REG_ANLPAR,&wANLPAR);

	if (pInfo->byRevId<REV_ID_VT6105_A0)
		byFlowCR=readb(&pMacRegs->byMISCCR0);
	else
		byFlowCR=readb(&pMacRegs->byFlowCR1);
	
	if (!(pInfo->mii_status & RHINE_DUPLEX_FULL)) {
		//Half duplex mode
		if (pInfo->byRevId<REV_ID_VT6105_A0) {
			byFlowCR&=~MISC_CR0_FDXRFEN;
		}
		else {
			byFlowCR&=~FLOWCR1_FDXTFCEN;
			byFlowCR&=~FLOWCR1_FDXRFCEN;			
		}
		
	} else {
		//Full duplxe mode		
		if (((wANAR & (ANAR_ASMDIR|ANAR_PAUSE))==ANAR_ASMDIR)
			&& ((wANLPAR & (ANLPAR_ASMDIR|ANLPAR_PAUSE))
			==(ANLPAR_ASMDIR|ANLPAR_PAUSE))) {				
			if (pInfo->byRevId<REV_ID_VT6105_A0) {
				byFlowCR&=~MISC_CR0_FDXRFEN;
			} 
			else {
				byFlowCR|=FLOWCR1_FDXTFCEN;
				byFlowCR&=~FLOWCR1_FDXRFCEN;	
			}								
		}	
		else if ((wANAR & ANAR_PAUSE) && (wANLPAR & ANLPAR_PAUSE)) {
			if (pInfo->byRevId<REV_ID_VT6105_A0) 
				byFlowCR|=MISC_CR0_FDXRFEN;
			else 
				byFlowCR|=(FLOWCR1_FDXTFCEN|FLOWCR1_FDXRFCEN);
		}
		else if (((wANAR & (ANAR_ASMDIR|ANAR_PAUSE))==(ANAR_ASMDIR|ANAR_PAUSE))
			&& ((wANLPAR & (ANLPAR_ASMDIR|ANLPAR_PAUSE)) ==ANLPAR_ASMDIR)) {
			if (pInfo->byRevId<REV_ID_VT6105_A0)
				byFlowCR|=MISC_CR0_FDXRFEN;
			else {
				byFlowCR&=~FLOWCR1_FDXTFCEN;
				byFlowCR|=FLOWCR1_FDXRFCEN;	
			}		
		}
		else {
			if (pInfo->byRevId<REV_ID_VT6105_A0) 
				byFlowCR&=~MISC_CR0_FDXRFEN;
			else 
				byFlowCR&=~(FLOWCR1_FDXTFCEN|FLOWCR1_FDXRFCEN);
		}

	}

	if (pInfo->byRevId< REV_ID_VT6105_A0) {
		byFlowCR &=~(MISC_CR0_HDXFEN|MISC_CR0_FDXTFEN); 
		writeb(byFlowCR,&pMacRegs->byMISCCR0);
	} 
	else {
		byFlowCR &=~FLOWCR1_HDXFCEN;
		writeb(byFlowCR,&pMacRegs->byFlowCR1);
	}

}
/***************************************************************************
*    ETHTOOL ioctl support routine
****************************************************************************/

#ifdef RHINE_ETHTOOL_IOCTL_SUPPORT
static int 	rhine_ethtool_ioctl(struct net_device* dev, struct ifreq* ifr) {
	struct ethtool_cmd	ecmd;
	PRHINE_INFO			pInfo=(PRHINE_INFO) dev->priv;
	PMAC_REGS			pMacRegs=pInfo->pMacRegs;
	int					flags;
	
	if (copy_from_user(&ecmd, ifr->ifr_data, sizeof(ecmd.cmd)))
        return -EFAULT;
        
    switch (ecmd.cmd) {
	case ETHTOOL_GSET: {
		U32	status;
		spin_lock_irqsave(&pInfo->lock,flags);
		status=rhine_check_media_mode(pInfo);
		spin_unlock_irqrestore(&pInfo->lock,flags);
		
		ecmd.supported=
			SUPPORTED_TP|SUPPORTED_Autoneg|SUPPORTED_10baseT_Half|SUPPORTED_10baseT_Full
			|SUPPORTED_100baseT_Half|SUPPORTED_100baseT_Full;
		
		if (status & RHINE_SPEED_100) 
			ecmd.speed=SPEED_100;
		else
			ecmd.speed=SPEED_10;
		
		ecmd.autoneg=(status & RHINE_AUTONEG_ENABLE) ? AUTONEG_ENABLE : AUTONEG_DISABLE;
		ecmd.port=PORT_TP;
		ecmd.transceiver=XCVR_INTERNAL;
		ecmd.phy_address=readb(&pMacRegs->byMIIAD) & 0x1F;
	
		if (status & RHINE_DUPLEX_FULL)
			ecmd.duplex=DUPLEX_FULL;
		else
			ecmd.duplex=DUPLEX_HALF;	
	
		if(copy_to_user(ifr->ifr_data, &ecmd, sizeof(ecmd)))
	           return -EFAULT;

		}		
		break;
		
	case ETHTOOL_SSET: {
		U32	curr_status;
		U32 new_status=0;
	    if (!capable(CAP_NET_ADMIN)){
	            return -EPERM;
	    }
	    
		if (BYTE_REG_BITS_IS_ON(CFGC_MEDEN, &pMacRegs->byCFGC)) {
			printk(KERN_INFO 
				"%s: The media mode have been forced by EEPROM utiltiy\n",dev->name);
			return -EPERM;    
		}
	
		if (copy_from_user(&ecmd, ifr->ifr_data, sizeof(ecmd)))
	           return -EFAULT;
			           
		curr_status=rhine_check_media_mode(pInfo);
		curr_status&=(~RHINE_LINK_FAIL);	
	
		new_status|=((ecmd.autoneg) ? RHINE_AUTONEG_ENABLE : 0);
		new_status|=((ecmd.speed==SPEED_100) ? RHINE_SPEED_100 : 0);
		new_status|=((ecmd.speed==SPEED_10) ? RHINE_SPEED_10 : 0);
		new_status|=((ecmd.duplex==DUPLEX_FULL) ? RHINE_DUPLEX_FULL : 0);
		
		if ((new_status & RHINE_AUTONEG_ENABLE) &&
			(new_status!=(curr_status| RHINE_AUTONEG_ENABLE)))
			return -EINVAL;
		spin_lock_irqsave(&pInfo->lock,flags);	
		rhine_set_media_mode(pInfo,new_status); 	
		spin_unlock_irqrestore(&pInfo->lock,flags);
		}
		break;
			
#ifdef ETHTOOL_GLINK
	case ETHTOOL_GLINK: {
		struct ethtool_value info;
		memset((void *) &info, 0, sizeof (info));
		info.cmd = ETHTOOL_GLINK;
		info.data = BYTE_REG_BITS_IS_ON(MIISR_LNKFL,&pMacRegs->byMIISR)
					? FALSE : TRUE;	

		if (copy_to_user(ifr->ifr_data, &info, sizeof (info)))
			return -EFAULT;	
		}
		break;
#endif

#ifdef ETHTOOL_GDRVINFO		
	case ETHTOOL_GDRVINFO:
		{
			struct ethtool_drvinfo info={ETHTOOL_GDRVINFO};
			strcpy(info.driver, RHINE_NAME);
			strcpy(info.version, RHINE_VERSION);
			strcpy(info.bus_info,pInfo->pcid->slot_name);
            if (copy_to_user(ifr->ifr_data, &info, sizeof(info)))
				return -EFAULT;                                           
    	}
    	break;
#endif     	

#ifdef ETHTOOL_GWOL
	case ETHTOOL_GWOL: {
	
		struct ethtool_wolinfo	wol={ETHTOOL_GWOL};
		memset(&wol,0,sizeof(wol));
		
		if (pInfo->byRevId>REV_ID_VT86C100A_E) {
			wol.supported=WAKE_PHY | WAKE_MAGIC | WAKE_UCAST | 
					WAKE_ARP;
			wol.wolopts|=WAKE_MAGIC;					
			if (pInfo->wol_opts & RHINE_WOL_PHY)
				wol.wolopts|=WAKE_PHY;
			if (pInfo->wol_opts & RHINE_WOL_UCAST)
				wol.wolopts|=WAKE_UCAST;
			if (pInfo->wol_opts & RHINE_WOL_ARP)			
				wol.wolopts|=WAKE_ARP;
				
			memcpy(&wol.sopass,pInfo->wol_passwd,6);
			
		} else
			return -EFAULT;
		
		if (copy_to_user(ifr->ifr_data, &wol,sizeof(wol)))
			return -EFAULT;	
		}
		break;
#endif

#ifdef ETHTOOL_SWOL
	case ETHTOOL_SWOL: {	
		struct ethtool_wolinfo	wol;
		
		if (copy_from_user(&wol, ifr->ifr_data, sizeof(wol))) 
			return -EFAULT;
				
		if (!(wol.wolopts & (WAKE_PHY|WAKE_MAGIC|WAKE_UCAST|WAKE_ARP)))
			return -EFAULT;						
		pInfo->wol_opts=RHINE_WOL_MAGIC; 		
		if (pInfo->byRevId>REV_ID_VT86C100A_E) {
			if (wol.wolopts & WAKE_PHY) {
				pInfo->wol_opts|=RHINE_WOL_PHY;
				pInfo->flags |=RHINE_FLAGS_WOL_ENABLED;
			}				
			if (wol.wolopts & WAKE_MAGIC) {
				pInfo->wol_opts|=RHINE_WOL_MAGIC;			
				pInfo->flags |=RHINE_FLAGS_WOL_ENABLED;
			}						
			if (wol.wolopts & WAKE_UCAST) {
				pInfo->wol_opts|=RHINE_WOL_UCAST;			
				pInfo->flags |=RHINE_FLAGS_WOL_ENABLED;
			}
			if (wol.wolopts & WAKE_ARP) {
				pInfo->wol_opts|=RHINE_WOL_ARP;			
				pInfo->flags |=RHINE_FLAGS_WOL_ENABLED;
			}						
			memcpy(pInfo->wol_passwd,wol.sopass,6);
		}
		else
			return -EFAULT;
			
		if (copy_to_user(ifr->ifr_data, &wol,sizeof(wol)))
			return -EFAULT;	
	}
	break;
#endif

	
#ifdef ETHTOOL_GMSGLVL
	case ETHTOOL_GMSGLVL: {
		struct ethtool_value edata={ETHTOOL_GMSGLVL};
		edata.data=msglevel;
		if (copy_to_user(ifr->ifr_data, &edata,sizeof(edata)))
			return -EFAULT;	
		}
		break;
#endif

#ifdef ETHTOOL_SMSGLVL
	case ETHTOOL_SMSGLVL: {
		struct ethtool_value edata={ETHTOOL_SMSGLVL};
		if (copy_from_user(&edata, ifr->ifr_data, sizeof(edata)))
			return -EFAULT;	
		msglevel=edata.data;
		}
		break;
#endif
	default:
		return -EOPNOTSUPP;
    }        
	return 0;
}

#endif

/***************************************************************************
*    MII ioctl support routine
****************************************************************************/
#ifdef RHINE_MII_IOCTL_SUPPORT
static int 	rhine_mii_ioctl(struct net_device* dev, struct ifreq* ifr,int cmd) {
	PRHINE_INFO	pInfo=(PRHINE_INFO) dev->priv;
	PMAC_REGS		pMacRegs=pInfo->pMacRegs;
	int				flags;
	struct mii_ioctl_data* pMiiData=(struct mii_ioctl_data*) &(ifr->ifr_data);

	switch(cmd) {
	case SIOCGMIIPHY:
		pMiiData->phy_id=readb(&pMacRegs->byMIIAD) & 0x1f;
		break;
	
	case SIOCGMIIREG:
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;		                 
		spin_lock_irqsave(&pInfo->lock,flags);
		rhine_mii_read(pInfo,pMiiData->reg_num & 0x1f, &(pMiiData->val_out));
		spin_unlock_irqrestore(&pInfo->lock,flags);
		break;
	
	case SIOCSMIIREG:
		if (!capable(CAP_NET_ADMIN))
			return -EPERM;
		spin_lock_irqsave(&pInfo->lock,flags);	
		rhine_mii_write(pInfo,pMiiData->reg_num & 0x1f,pMiiData->val_in);
		rhine_check_media_mode(pInfo);
		spin_unlock_irqrestore(&pInfo->lock,flags);				
		break;
	default:
          return -EOPNOTSUPP;
	}
	return 0;
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)

#ifdef CONFIG_PM
static int
rhine_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
	struct pci_dev *pdev = NULL;
	switch(event) {
	case SYS_DOWN:
	case SYS_HALT:
	case SYS_POWER_OFF:
       	pci_for_each_dev(pdev) {
       		if(pci_dev_driver(pdev) == &rhine_driver) {
       			if (pci_get_drvdata(pdev))
					rhine_suspend(pdev, 3);
			}
		}
	}
	return NOTIFY_DONE;
}

static int
rhine_suspend(struct pci_dev *pcid, u32 state)
{
	PRHINE_INFO pInfo=pci_get_drvdata(pcid);	
	netif_stop_queue(pInfo->dev);	
	spin_lock_irq(&pInfo->lock);
	
	pci_save_state(pcid, pInfo->pci_state);

#ifdef ETHTOOL_GWOL
		rhine_shutdown(pInfo);			
	if (pInfo->flags & RHINE_FLAGS_WOL_ENABLED) {
		rhine_get_ip(pInfo);
		rhine_save_context(pInfo,&pInfo->context);				
		rhine_set_wol(pInfo);		
		pci_enable_wake(pcid, 3, 1);
		pci_set_power_state(pcid, 3);
	} else {
		rhine_save_context(pInfo,&pInfo->context);	
		pci_disable_device(pcid);
		pci_set_power_state(pcid, state);
	}
#else
	pci_disable_device(pcid);
	pci_set_power_state(pcid, state);
#endif
	spin_unlock_irq(&pInfo->lock);
	return 0;
}

static int
rhine_resume(struct pci_dev *pcid)
{
	PRHINE_INFO pInfo=pci_get_drvdata(pcid);
	
	pci_set_power_state(pcid, 0);
	pci_enable_wake(pcid, 0, 0);
	pci_restore_state(pcid, pInfo->pci_state);	
	
	enable_mmio(pInfo);
	
	rhine_wol_reset(pInfo);	
	
	if (netif_running(pInfo->dev)) {
		spin_lock_irq(&pInfo->lock);
		rhine_restore_context(pInfo,&pInfo->context);
		rhine_init_registers(pInfo, RHINE_INIT_WOL);
		spin_unlock_irq(&pInfo->lock);
		rhine_tx_srv(pInfo,0);
		rhine_enable_int(pInfo);
	}
	
	return 0;
}

static int 
rhine_netdev_event(struct notifier_block *nb,
	unsigned long notification, void *ptr) {
	struct in_ifaddr* ifa=(struct in_ifaddr*) ptr;
	struct net_device*	dev;
	PRHINE_INFO	pInfo;	

	if (ifa)  {
		dev=ifa->ifa_dev->dev;
		
		for (pInfo=pRhine3_Infos;pInfo;pInfo=pInfo->next) {
			if (pInfo->dev==dev) 
				rhine_get_ip(pInfo);
		}
	}
	return NOTIFY_DONE;
}
#endif /*CONFIG_PM*/

#endif /*VERSION >= 2.4.9*/

