#!/bin/sh
###############################################################################
# Part of Marvell Yukon/SysKonnect sk98lin Driver for Linux                   #
###############################################################################
# Installation script for Marvell Chip based Ethernet Gigabit Cards           #
# $Revision: 1.12 $                                                            #
# $Date: 2003/09/23 08:09:28 $                                                #
# =========================================================================== #
#                                                                             #
#  Main - Global function                                                     #
#                                                                             #
# Description:                                                                #
#  This file includes all functions and parts of the script                   #
#                                                                             #
# Returns:                                                                    #
#       N/A                                                                   #
# =========================================================================== #
# Usage:                                                                      #
#     ./install.sh                                                            #
#                                                                             #
# =========================================================================== #
# COPYRIGHT NOTICE :                                                          #
#                                                                             #
# (C)Copyright 1999-2003 Marvell(R).                                                              #
#                                                                             #
#  This program is free software; you can redistribute it                     # 
#  and/or modify it under the terms of the GNU General Public                 #
#  License as published by the Free Software Foundation; either               #
#  version 2 of the License, or (at your option) any later version.           #
#                                                                             #
#                                                                             #
# WARRANTY DISCLAIMER:                                                        #
#                                                                             #
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT WITHOUT #
# ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF MERCHANTABILITY OR       #
# FITNESS FOR A PARTICULAR PURPOSE.                                           #
#                                                                             #
# =========================================================================== #
#                                                                             #
#     History:                                                                #
#     2003-09-23    Version 1.08 - Better CPU and Comp.           (mlindner)  #
#                                  New function clean()                       #
#     2003-08-21    Version 1.07 - "silent mode" changes          (mlindner)  #
#     2003-08-21    Version 1.06 - Added "silent mode"            (mlindner)  #
#                                  Added secure temp creation                 #
#                                  Better cleanup functions                   #
#                                  Better CPU and Arch detection              #
#                                  Parameter check                            #
#                                  autoconf.h management                      #
#     2003-08-06    Version 1.05 - Small fixes                    (mlindner)  #
#     2003-06-03    Version 1.04 - Additions for SK_SLIM          (mlindner)  #
#     2003-03-19    Version 1.03 - New driver error descriptions  (mlindner)  #
#     2003-02-13    Version 1.02 - Install failure checks         (mlindner)  #
#                                  Disable CONFIG_MODVERSIONS                 #
#                                  Improve script logging                     #
#                                  Add "working" messages                     #
#     2003-02-07    Version 1.01 - Editorial changes              (mlindner)  #
#     2003-01-30    Version 1.00 - Initial release                (mlindner)  #
#                                                                             #
#                                                                             #
###############################################################################


# Functions
###########################################

function message_status ()
{
	# Print a status message 
	# Syntax: message_status STATUS STRING
	# 	STATUS
	#		0) FAILED
	#		1) OK
	#		2) WARN
	# Author: mlindner
	# Returns:
	#       N/A

	if test -z "$LINES" -o -z "$COLUMNS" ; then
		eval `stty size 2>/dev/null | (read L C; \
		echo LINES=${L:-24} COLUMNS=${C:-80})`
	fi
	test $COLUMNS -eq 0 && COLUMNS=80
	esc=`echo -en "\033"`
	error="${esc}[1;31m"
	ok="${esc}[1;32m"
	warn="${esc}[1;33m"
	working="${esc}[1;34m"
	stat=`echo -en "\015${esc}[${COLUMNS}C${esc}[10D"`
	norm=`echo -en "${esc}[m\017"`

	m_ok="${stat}${ok}passed ${norm}"
	m_failed="${stat}${error}failed ${norm}"
	m_warning="${stat}${warn}warning${norm}"
	m_working="${stat}${working}working${norm}"
	if [ "$2" != "working" ]; then
		echo -n " ($2)"
	fi

	case "$1" in
	3)	echo -n $m_working ;;
	2)	echo $m_warning ;;
	1)	echo $m_ok ;;
	0)	echo $m_failed ;;
	esac

	return 0;
}

function make_safe_tmp_dir ()
{
	# Generate safe tmp dir
	# Syntax: make_safe_tmp_dir
	# Author: mlindner
	# Returns:
	#       TMP_DIRECTORY

	fname="Create tmp dir"
	echo -n $fname
	message_status 3 "working"
	BASE_TMP_DIR="/tmp"
	if ! [ -e "$BASE_TMP_DIR" ]; then
		BASE_TMP_DIR=`pwd`
		mkdir ${BASE_TMP_DIR}/tmp
		BASE_TMP_DIR=`echo ${BASE_TMP_DIR}/tmp`
	fi	

	TMP_DIR=${BASE_TMP_DIR}/Sk98I`awk 'BEGIN { srand(); for (i=1;i<21;i++) { a=95; while (a > 90 && a < 97) { a=65+int(50*rand())}; printf("%c", a) } }'`

	[ -e "$TMP_DIR" ] && rm -rf $TMP_DIR
	if [ -e "$TMP_DIR" ]; then 
		echo
		echo "My temp dir exists already."
		echo "This looks like a symlink attack!"
		echo 
		echo "*** Aborting"
		echo
		exit 1
	fi

	if [ "$TMP_DIR" = "$BASE_TMP_DIR" -o "$TMP_DIR" = "/" ]; then
		echo 
		echo "\"$TMP_DIR\" is an unacceptable value for the temp dir. Please"
		echo "edit the variable definition for $TMP_DIR and try again."
		echo
		echo "*** Aborting"
		echo
		exit 1
	fi

	mkdir $TMP_DIR
	chmod 700 $TMP_DIR &> /dev/null

	echo -en "\015"
	echo -n $fname
	message_status 1 "$TMP_DIR"
}

function cleanup ()
{
	# Restore all files
	# Syntax: cleanup
	# Author: mlindner
	# Returns:
	#       N/A

	
	# Restore version.h
	version_h_management 0

	# Restore old autoconf.h file
	autoconf_h_management 0

	if [ -e ${TMP_DIR}/config ]; then
		if [ ${KERNEL_SOURCE} ]; then
			cp ${TMP_DIR}/config ${KERNEL_SOURCE}/.config
		fi
	fi

	return
}

function clean ()
{
        # Clean temp directory
        # Syntax: clean
        # Author: mlindner
        # Returns:
        #       N/A

	echo -n "Delete temp directories"
 	cd $working_dir
	if [ ${TMP_DIR} ]; then
 		rm -rf ${TMP_DIR}
	else
		rm -rf /tmp/Sk98I*
	fi
 	rm -rf $drv_name.o
	message_status 1 "done"

	return
}


function inst_failed ()
{
	# Print a message and exit 
	# Syntax: inst_failed MESSAGE
	# Author: mlindner
	# Returns:
	#       N/A

	inst_failed="Installation of $drv_name driver module failed."
	echo
	echo $1
	echo
	echo $inst_failed

	cleanup	# Cleanup all tmp files and exit
	clean
	exit 1

}


function header_check_failed ()
{
	# Print a error message and exit 
	# Syntax: header_check_failed
	# Author: mlindner
	# Returns:
	#       N/A

	echo "There is a mismatch between the current running kernel and"
	echo "the header files the kernel module will be compiled with."
	echo " "
	echo "For instance, it might be, that you run kernel version"
	echo "2.4.20, but the header files the kernel module will be"
	echo "compiled with refer to kernel version 2.4.21"
	echo 
	echo "Due to this mismatch, you will not be able to load the "
	echo "driver without the force option (insmod -f $drv_name) after"
	echo "its compilation finished."
	echo " "
	echo "This problem can be resolved by overwriting the current"
	echo "include/version.h (which corresponds to another kernel "
	echo "version), with the include/version.h of the kernel version"
	echo "currently running."
	echo " "
 	echo "BEWARE: OVERWRITE THE FILE ONLY IF YOU HAVE REALLY THE "
	echo "CORRECT HEADER FILE CORRESPONDING TO THE CURRENT RUNNING"
	echo " "
	echo "If you don't have the same kernel version, please install  "; \
	echo "the sources or a new kernel. It's not possible to mix      "; \
	echo "different kernel versions!                                 "; \
	echo "                                                           "; \
	echo "If you know what you are doing and want to override this   "; \
	echo "check, you can do so by setting IGNORE_HEADER_MISMATCH     "; \
	echo "system variable:                                           "; \
	echo "    Example: export IGNORE_HEADER_MISMATCH="1"             "; \
	echo "                                                           "; \
	echo "or change the file ${KERNEL_HEADER}/linux/version.h,       "; \
	echo "remove the define UTS_RELEASE line and insert:             "; \
	echo "#define UTS_RELEASE \"${KERNEL_VERSION}\"                  "; \
	echo "                                                           "; \
	echo "    Your kernel version: ${KERNEL_VERSION}                 "; \
	echo "    Your header version: $header_version                   "; \
	echo $inst_failed
	echo "                                                           "; \
	cleanup
	clean
}


function create_makefile ()
{
	# Create makefile
	# Syntax: create_makefile
	# Author: mlindner
	# Returns:
	#       N/A

	{
	echo '# Makefile for Marvell Yukon/SysKonnect SK-98xx/SK-95xx Gigabit'
	echo '# Ethernet Adapter driver'
	echo ''
	echo '# just to get the CONFIG_SMP and CONFIG_MODVERSIONS defines:'
	echo 'ifeq ($(KERNEL_SOURCE)/.config, $(wildcard $(KERNEL_SOURCE)/.config))'
	echo 'include $(KERNEL_SOURCE)/.config'
	echo 'endif'
	echo 'SYSINC =  -I$(KERNEL_HEADER) -I.'
	echo 'SYSDEF = -DLINUX -D__KERNEL__'
	echo 'ifdef CONFIG_SMP'
	echo 'SYSDEF += -D__SMP__'
	echo 'endif'
	echo 'SRCDEF = -DMODULE -O2 -DSK_USE_CSUM -DGENESIS -DYUKON'
	echo 'ifdef CONFIG_MODVERSIONS'
	echo 'SRCDEF += -DMODVERSIONS -include $(KERNEL_HEADER)/linux/modversions.h'
	echo 'endif'
	echo 'USERDEF='
	echo 'WARNDEF=-Wall -Wimplicit -Wreturn-type -Wswitch -Wformat -Wchar-subscripts \'
	echo '	   -Wparentheses -Wpointer-arith -Wcast-qual -Wno-multichar  \'
	echo '	   -Wno-cast-qual'
	echo 'INCLUDE= $(SYSINC)'
	echo 'DEFINES=  $(SYSDEF) $(SRCDEF) $(USERDEF) $(WARNDEF)'
	echo 'SRCFILES = skge.c skgeinit.c skgesirq.c skxmac2.c skvpd.c skgehwt.c \'
	echo '	   skqueue.c sktimer.c ski2c.c sklm80.c skrlmt.c skgepnmi.c \'
	echo '	   skaddr.c skcsum.c skproc.c skdim.c'
	echo 'OBJECTS =  skge.o skaddr.o skgehwt.o skgeinit.o skgepnmi.o skgesirq.o \'
	echo '	   ski2c.o sklm80.o skqueue.o skrlmt.o sktimer.o skvpd.o skdim.o\'
	echo '	   skxmac2.o skcsum.o skproc.o'
	echo 'DRVBIN = sk98lin.o'
	echo 'LD	= ld'
	echo 'CC	= gcc'
	echo 'CFLAGS	= $(INCLUDE) $(DEFINES)'
	echo 'FILES	= $(SRCFILES) makefile'
	echo 'TARGETS	= $(DRVBIN)'
	echo '.c.o:   $<'
	echo '	$(CC) $(CFLAGS) -c $<'
	echo 'all:  $(OBJECTS)'
	echo '	$(LD) -r -o $(DRVBIN) $(OBJECTS)'
	echo 'clean:'
	echo '	rm *.o'
	echo '*.o: \'
	echo '	h/*.h'
	} &> ${TMP_DIR}/Makefile

}


function read_and_change_version_file ()
{
	# Check version.h
	# Syntax: read_and_change_version_file FILE_AS_STRING
	# Author: mlindner
	# Returns:
	#       N/A

	dline_count=0
	rm -rf ${TMP_DIR}/newversion.h
	pure_kernel_version=`echo ${KERNEL_VERSION}`
	pure_kernel_version=`echo $pure_kernel_version | cut -d '-' -f 1`

	while read line  # For as many lines as the input file has...
	do
		line_count=`echo $line | grep -c "UTS_RELEASE"`
		if [ $line_count != 0 ]; then
			if [ $dline_count == 0 ]; then
				version_count=`echo $line | grep -c "$pure_kernel_version"`
				if [ $version_count == 1 ]; then
				# Change line
					echo "#define UTS_RELEASE \"${KERNEL_VERSION}\"" \
						>> ${TMP_DIR}/newversion.h
					dline_count=1
				fi
			fi
		else
			echo "$line" >> ${TMP_DIR}/newversion.h
		fi
	done

	if [ $dline_count == 0 ]; then
		header_check_failed
	fi

	cp ${TMP_DIR}/newversion.h ${KERNEL_HEADER}/linux/version.h
}


function start_sequence ()
{
	# Print copyright informations, mode selection and check
	# Syntax: start_sequence
	# Author: mlindner
	# Returns:
	#       N/A

	# Start. Choose a installation method and check method
	echo 
	echo "Installation script for $drv_name driver."
	echo "Version $VERSION"
	echo "(C)Copyright 1999-2003 Marvell(R)."
	echo "===================================================="
	echo "Add to your trouble-report the logfile install.log"
	echo "which is located in the  DriverInstall directory."
	echo "===================================================="
	echo 

	# Silent option. Return...
	[ "$OPTION_SILENT" ] && return


	PS3='Choose your favorite installation method: ' # Sets the prompt string.

	echo

	select user_sel in "user" "expert"
	do
		break  # if no 'break' here, keeps looping forever.
	done

	clear

	if [ "$user_sel" == "user" ]
	then
		echo "Please read this carfully!"
		echo
		echo "This script will automatically compile and load the $drv_name"
		echo "driver on your host system. Before performing both compilation"
		echo "and loading, it is necessary to shutdown any device using the" 
		echo "$drv_name kernel module and to unload the old $drv_name kernel "
		echo "module. This script will do this automatically per default."
		echo "If you want to shutdown and unload the old $drv_name kernel module"
		echo "manually, run the script in the EXPERT mode."
		echo " "
		echo "Please plug a card into your machine. Without a card we aren't"
		echo "able to check the full driver functionality."
		echo
		echo -n "Do you want proceed? (y/N) "

		old_tty_settings=$(stty -g)		# Save old settings.
		stty -icanon
		Keypress=$(head -c1)
		stty "$old_tty_settings"			# Restore old terminal settings.
		echo "+++ Install mode: User" >> $logfile 2>&1

		if [ "$Keypress" == "Y" ]
		then
			clear
		else
			if [ "$Keypress" == "y" ]
			then
				clear
			else
				echo "Exit"
				cleanup
				clean
				exit 0
			fi
		fi
		
		export REMOVE_SKDEVICE=1

	else
		echo "+++ Install mode: Expert" >> $logfile 2>&1
		clear
	fi
}


function check_user_and_tools ()
{
	# Check user informations and the existence of defferent tools
	# Syntax:  check_user_and_tools
	# Author: mlindner
	# Returns:
	#       N/A

	# Check user id
	inst_failed="Installation of $drv_name driver module failed."
	fname="Check user id"
	echo -n $fname
	if [ `id -u` != 0 ]; then
		message_status 0 `id -u`
		echo "+++ Wrong user"  >> $logfile 2>&1
		echo "You must have root privileges to install the $drv_name driver module."
		inst_failed 
	else
		message_status 1 `id -u`
	fi

	# Check machine type
	echo -n "Check host type"
	smp_count=`cat /proc/cpuinfo | grep processor -c`
	if [ $smp_count -gt 1 ]; then
		smp_count=1
	else
		smp_count=0
	fi


	kernel_machine=`uname -m`
	message_status 1 "done"
	echo "+++ smp_count=$smp_count" >> $logfile 2>&1
	echo "+++ kernel_machine=$kernel_machine" >> $logfile 2>&1


	# Check gcc
	echo -n "Check compiler"
	if [ `which gcc` ]; then
		message_status 1 `which gcc`
	else
		message_status 0 "not found"
		echo "+++ Compiler not found" >> $logfile 2>&1	
		inst_failed "You have to install the gcc compiler."
	fi


	# Check module support
	echo -n "Check module support"
	if [ `which insmod` ]; then
		insmod_bin=`which insmod`
		message_status 1 $insmod_bin
	else
		if [ -e /sbin/insmod ]; then
			insmod_bin="/sbin/insmod"
			message_status 1 $insmod_bin
		else
			if [ -e /usr/sbin/insmod ]; then
				insmod_bin=`echo "/usr/sbin/insmod"`
				message_status 1 $insmod_bin
			else
				message_status 0 "not found"
				echo "+++ Insmod not found" >> $logfile 2>&1	
				inst_failed "You have to install the modutils package."
			fi
		fi
	fi


	# Check make 
	echo -n "Check make"
	if [ `which make` ]; then
		message_status 1 `which make`
	else
		message_status 0 "not found"
		echo "+++ Make not found" >> $logfile 2>&1	
		inst_failed "You have to install the make package."
	fi


	# Check archive file
	echo -n "Check archive file"
	if [ -e $drv_name.tar.bz2 ]; then
		message_status 1 "$drv_name"
	else
		message_status 0 "not found"
 		echo "You have to copy the $drv_name.tar.bz2 file into the current"
		echo "$working_dir directory"
		echo "+++ $drv_name.tar.bz package not found" >> $logfile 2>&1	
		echo $inst_failed
		cleanup
		clean
		exit
	fi

}


function kernel_check_failed ()
{
	# Print kernel error informations
	# Syntax:  kernel_check_failed
	# Author: mlindner
	# Returns:

	echo "Kernel version unsupported."
	echo "This driver was developed for the kernel family 2.4.x"
	echo "higher then 2.4.13. If you are still using a old version"
	echo "of the kernel, please uptade to the newest version from"
	echo "ftp://ftp.kernel.org"
	echo "+++ Kernel version unsupported" >> $logfile 2>&1	
	echo $inst_failed
	echo
	cleanup
	clean
	exit
}


function check_kernel_informations ()
{
	# Check kernel and module informations
	# Syntax:  check_kernel_informations
	# Author: mlindner
	# Returns:
	#       N/A

	# Check kernel version
	export KERNEL_VERSION=`uname -r`
	echo -n "Check kernel version"
	echo "+++ Kernel version ${KERNEL_VERSION}" >> $logfile 2>&1
	
	split_kernel_ver=`echo ${KERNEL_VERSION} | cut -d '.' -f 1`	
	if [ $split_kernel_ver != 2 ]; then
		message_status 0 ${KERNEL_VERSION}
		kernel_check_failed
	fi
	KERNEL_FAMILY="$split_kernel_ver"

	split_kernel_ver=`echo ${KERNEL_VERSION} | cut -d '.' -f 2`	
	if [ $split_kernel_ver != 4 ]; then
		message_status 0 ${KERNEL_VERSION}
		kernel_check_failed
	fi
	KERNEL_FAMILY="$KERNEL_FAMILY.$split_kernel_ver"

	split_kernel_ver=`echo ${KERNEL_VERSION} | cut -d '.' -f 3`	
	split_kernel_ver2=`echo $split_kernel_ver | cut -d '-' -f 1`	
	if [ "$split_kernel_ver2" -lt  13 ]; then
		message_status 0 ${KERNEL_VERSION}
		kernel_check_failed
	fi
	message_status 1 ${KERNEL_VERSION}

	# Check gcc and kernel version
	gcc_version=`gcc -v 2>&1 | tail -1`
	gcc_version=`echo $gcc_version | cut -d ' ' -f 3`
	kernel_gcc_version=`cat /proc/version | sed -e "s/^.*gcc version//g"`
	kernel_gcc_version=`echo $kernel_gcc_version | sed -e "s/)//g"`
	kernel_gcc_version=`echo $kernel_gcc_version | cut -d ' ' -f 1`
	echo -n "Check kernel gcc version (${kernel_gcc_version})"

	if [ $kernel_gcc_version != $gcc_version ]; then
		message_status 0 "Kernel:$kernel_gcc_version != gcc:$gcc_version"
		echo "+++ Mismatch!!! Kernel:$kernel_gcc_version != gcc:$gcc_version" >> $logfile 2>&1
		if [ -z ${IGNORE_CC_MISMATCH} ]; then \

		echo "There is a version mismatch between the compiler that was used"
		echo "to build the current running kernel and the compiler which you"
		echo "intend to compile the kernel module with. In most of the cases,"
		echo "this is no problem, but there are cases in which this compiler-"
		echo "mismatch leads to unexpected system crashes"
		echo " "
		echo "If you know what you are doing and want to override this   "; \
		echo "check, you can do so by setting IGNORE_CC_MISMATCH system  "; \
		echo "variable:                                                  "; \
		echo "    Example: export IGNORE_CC_MISMATCH="1"                 "; \
		echo $inst_failed
		cleanup
		clean
		exit 1
		fi
	else
		message_status 1 "Kernel:$kernel_gcc_version == gcc:$gcc_version"
	fi


	# Check the driver availability
	echo -n "Check $drv_name driver availability"
	check_sk98lin=`lsmod | grep $drv_name -c` 
	if [ $check_sk98lin != 1 ]; then
		message_status 1 "not loaded"
	else
		if [ -z ${IGNORE_SKAVAIL_CHECK} ]; then 
			if [ "$user_sel" == "user" ]
			then
				message_status 1 "loaded"
			else
				echo "+++ Driver loaded. OK" >> $logfile 2>&1
				message_status 0 "loaded"
			fi

			if [ -z ${REMOVE_SKDEVICE} ]; then 
			echo
			echo "Driver $drv_name loaded. Please remove the driver with rmmod."
			echo "If you want override this check, you can do so by setting   "
			echo "IGNORE_SKAVAIL_CHECK:"; 
			echo "    Example: export IGNORE_SKAVAIL_CHECK="1"             "
			echo
			echo "If you want to remove the devices and the driver automatically,"
			echo "you can do so by setting REMOVE_SKDEVICE:"
			echo "    Example: export REMOVE_SKDEVICE="1"             "
			echo
			echo "+++ Driver loaded. ERROR!" >> $logfile 2>&1
			echo $inst_failed
			cleanup
			clean
			exit 1
			else
				echo -n "Disconnect devices: "
				for devices in `ls /proc/net/sk98lin`; do
					echo -n "$devices "
					ifconfig $devices down &> /dev/null
				done
				message_status 1 "done"
				echo -n "Remove driver"
				rmmod $drv_name &> /dev/null
				message_status 1 "done"
			fi
		else
			message_status 2 "loaded"
		fi
	fi


	# Check header files
	echo -n "Check kernel header files"
	if [ -d /usr/src/linux/include/linux/ ]; then
		message_status 1 "/usr/src/linux"
		export KERNEL_HEADER="/usr/src/linux/include";
		export KERNEL_SOURCE="/usr/src/linux";
	else
		if [ -d /usr/src/linux-${KERNEL_VERSION}/include/linux/ ]; then
			message_status 1 "/usr/src/linux-${KERNEL_VERSION}"
			export KERNEL_HEADER="/usr/src/linux-${KERNEL_VERSION}/include";
			export KERNEL_SOURCE="/usr/src/linux-${KERNEL_VERSION}";
		else
			kernel_check_dir="linux-$KERNEL_FAMILY"
			if [ -d /usr/src/$kernel_check_dir/include/linux/ ]; then
				message_status 1 "/usr/src/$kernel_check_dir"
				export KERNEL_HEADER="/usr/src/$kernel_check_dir/include";
				export KERNEL_SOURCE="/usr/src/$kernel_check_dir";
			else			
			message_status 0 "not found"
			echo "Kernel header not found. Please install the linux header files "
			echo "development package or crate a symbolic link from the "
			echo "/usr/src/KERNEL_VERSION directory to linux"
			echo "     Example: ln -s /usr/src/KERNEL_VERSION /usr/src/linux"
			echo ""
			echo "+++ Kernel header not found. ln -s /usr/src/KERNEL_VERSION?" >> $logfile 2>&1
			echo $inst_failed
			cleanup
			clean
			exit 1
			fi
   		fi
	fi
}


function unpack_driver ()
{
	# Create tmp dir and unpack the driver
	# Syntax: unpack_driver 
	# Author: mlindner
	# Returns:
	#       N/A


	# Create tmp dir and unpack the driver
	fname="Unpack the sources"
	echo -n $fname
	message_status 3 "working"

	echo "+++ Unpack the sources" >> $logfile
	echo "+++ ====================================" >> $logfile
	cd $working_dir
	cp $drv_name.tar.bz2 ${TMP_DIR}/

	cd ${TMP_DIR}
	bunzip2 $drv_name.tar.bz2 &> /dev/null
	echo "+++ tar xfv $drv_name.tar" >> $logfile
	tar xfv $drv_name.tar >> $logfile
	cd ..
	if [ -f ${TMP_DIR}/skge.c ]; then
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"
	else
		echo -en "\015"
		echo -n $fname
		message_status 0 "error"
	 	echo
		echo "An error has occurred during the unpack proces which prevented "; \
		echo "the installation from completing.                              "; \
		echo "Take a look at the log file install.log for more informations.  "; \
		echo "+++ Unpack error!!!" >> $logfile 2>&1
		echo $inst_failed
		cleanup
		clean
		exit 1
	fi
}


function check_config ()
{
	# Check and change the config file
	# Syntax: check_config
	# Author: mlindner
	# Returns:
	#       N/A

	# Backup old .config file
	fname="Copy and check .config file"
	echo -n $fname
	message_status 3 "working"

	# Check config
	cp ${KERNEL_SOURCE}/.config ${TMP_DIR}/config
	rm -rf ${TMP_DIR}/newconfig &> /dev/null
	if [ $smp_count == 1 ]; then
		sed -e 's/# CONFIG_SMP is not set/CONFIG_SMP=y/' \
			${TMP_DIR}/config \
			>> ${TMP_DIR}/newconfig
	else
		sed -e 's/CONFIG_SMP=y/# CONFIG_SMP is not set/' \
			${TMP_DIR}/config >> ${TMP_DIR}/newconfig
	fi

	# Set highmem back
	sed -e 's/CONFIG_HIGHMEM4G=y/# CONFIG_HIGHMEM4G is not set/' \
		${TMP_DIR}/newconfig &> ${TMP_DIR}/newconfig2
	sed -e 's/CONFIG_HIGHMEM64G=y/# CONFIG_HIGHMEM64G is not set/' \
		${TMP_DIR}/newconfig2 &> ${TMP_DIR}/newconfig
	sed -e 's/# CONFIG_NOHIGHMEM is not set/CONFIG_NOHIGHMEM=y/' \
		${TMP_DIR}/newconfig &> ${TMP_DIR}/newconfig2
	sed -e 's/CONFIG_HIGHMEM=y/# CONFIG_HIGHMEM is not set/' \
		${TMP_DIR}/newconfig2 &> ${TMP_DIR}/newconfig
			
	# Set version management back
	sed -e 's/CONFIG_MODVERSIONS=y/# CONFIG_MODVERSIONS is not set/' \
		${TMP_DIR}/newconfig &> ${TMP_DIR}/newconfig2

	
	mv ${TMP_DIR}/newconfig2 ${TMP_DIR}/newconfig

	cp ${TMP_DIR}/newconfig ${KERNEL_SOURCE}/.config

	echo -en "\015"
	echo -n $fname
	message_status 1 "done"

}


function generate_config ()
{
	# Check config file; Save old version and modify
	# Syntax: generate_config
	# Author: mlindner
	# Returns:
	#       N/A

	echo -n "Check sources for .config file"
	echo 'while true; do echo -en "\012";done' &> ${TMP_DIR}/enter
	chmod 755 ${TMP_DIR}/enter

	if [ -f ${KERNEL_SOURCE}/.config ]; then
		message_status 1 "${KERNEL_SOURCE}/.config"
		##
		# Sources umkopieren in /usr/src/linux und make ausfuehren
		# Davor checken ob sk98lin-treiber bereits gebaut wurde und
		# wenn ja, dann make menuconfig aufrufen unter /usr/src/linux
		##
		if [ -f ${KERNEL_SOURCE}/drivers/net/$drv_name/$drv_name.o ]; then
			rm -rf ${KERNEL_SOURCE}/drivers/net/sk98lin/*.o
			KERNEL_TREE_MAKE=1
		else
			check_config

			fname="Execute: make config"
			echo -n $fname
			message_status 3 "working"
			cd ${KERNEL_SOURCE}
			${TMP_DIR}/enter | make config &> /dev/null
			echo -en "\015"
			echo -n $fname
			message_status 1 "done"

			# Make dep
			fname="Execute: make dep"
			echo -n $fname
			message_status 3 "working"
			cd ${KERNEL_SOURCE}
			make dep &> /dev/null
			echo -en "\015"
			echo -n $fname
			message_status 1 "done"
		fi
	else
		message_status 1 "none"	
		fname="Execute: make mrproper"
		echo -n $fname
		message_status 3 "working"
		cd ${KERNEL_SOURCE}
		make mrproper &> /dev/null 
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"

		if [ -d ${KERNEL_SOURCE}/configs ]; then
			echo -n "Config files found"
			message_status 1 "${KERNEL_SOURCE}/configs/"
			echo -n "Copying file from config directory"
			cpu_count=`cat /proc/cpuinfo | grep "cpu family" -c`

			pure_kernel=`echo ${KERNEL_VERSION} | cut -d '-' -f 1`
			mach_config="${KERNEL_SOURCE}/configs/kernel-$pure_kernel-$kernel_machine.config"

			if [ -f $mach_config ]; then
 				if [ $cpu_count != 1 ]; then
					if [ $cpu_count -gt 1 ]; then
						mach_config="${KERNEL_SOURCE}/configs/kernel-$pure_kernel-$kernel_machine-smp.config"
						cp $mach_config ${KERNEL_SOURCE}/.config
						message_status 1 "done"
					else
						message_status 1 "file not found"
					fi
				else
					cp $mach_config ${KERNEL_SOURCE}/.config
					message_status 1 "done"
				fi
			else
				message_status 1 "file not found"
			fi
		fi


		if [ -f ${TMP_DIR}/version.h ]; then
			echo -n "Regenerate version.h"
			cp ${TMP_DIR}/version.h ${KERNEL_SOURCE}/include/linux/
			message_status 1 "done"
		fi

		fname="Create .config file"
		echo -n $fname
		message_status 3 "working"
		${TMP_DIR}/enter | make config &> /dev/null
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"

		check_config

		fname="Execute: make config"
		echo -n $fname
		message_status 3 "working"
		${TMP_DIR}/enter | make config &> /dev/null
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"

		# Make dep
		fname="Execute: make dep"
		echo -n $fname
		message_status 3 "working"
		make dep &> /dev/null
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"

	fi
}


function make_driver ()
{
	# Configure, check and build the driver
	# Syntax: make_driver
	# Author: mlindner
	# Returns:
	#       N/A

	# Check header version
	echo -n "Check kernel header version"
	header_count=`grep -c RELEASE ${KERNEL_HEADER}/linux/version.h`

	if [ $header_count != 1 ]; then
		message_status 2 "not recognized"
	else	
		header_version=`grep RELEASE ${KERNEL_HEADER}/linux/version.h`
		header_version=`echo $header_version | cut -d '"' -f 2`
		if [ $header_version != ${KERNEL_VERSION} ]; then
			if [ -z ${IGNORE_HEADER_MISMATCH} ]; then
				if [ "$user_sel" == "user" ]
				then
					message_status 1 "Kernel:${KERNEL_VERSION} != Header:$header_version"
					echo -n "Save old version.h"
					cp ${KERNEL_HEADER}/linux/version.h ./
					message_status 1 "saved"	

					echo -n "Check and create new version.h"
					tmp_version=`cat ${KERNEL_HEADER}/linux/version.h`
					read_and_change_version_file <<INPUTSTART
$tmp_version
INPUTSTART

					message_status 1 "created"
				else
					message_status 0 "Kernel:${KERNEL_VERSION} != Header:$header_version"
					echo "+++ Kernel:${KERNEL_VERSION} != Header:$header_version" >> $logfile 2>&1
					header_check_failed
				fi
			fi
		else
			message_status 1 "Kernel:${KERNEL_VERSION} == Header:$header_version"
		fi
	fi

	
	# Compile the driver
	fname="Compile the driver"
	echo -n $fname
	message_status 3 "working"

	echo >> $logfile 2>&1 
	echo "+++ Compile the driver" >> $logfile 2>&1
	echo "+++ ====================================" >> $logfile 2>&1
	cd ${TMP_DIR}

	make >> $logfile 2>&1

	if [ -f $drv_name.o ]; then
		cp $drv_name.o ../
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"
	else
		echo -en "\015"
		echo -n $fname
		message_status 0 "error"
 		echo
		echo "An error has occurred during the compile proces which prevented "; \
		echo "the installation from completing.                              "; \
		echo "Take a look at the log file install.log for more informations.  "; \
		echo "+++ Compiler error" >> $logfile 2>&1
		echo $inst_failed
		cleanup
		clean
		exit 1
	fi
}


function extract_params ()
{
	# Extract all given parameters
	# Syntax:  extract_params
	# Author: mlindner
	# Returns:
	#       N/A

	if [ $# -eq "0" ];then
		# Script invoked with no command-line args?
		return
	fi

	while getopts ":sc" Option
		do
  		case $Option in
			s     ) OPTION_SILENT=1;;
			c     ) OPTION_CLEANUP=1;;
			*     ) echo "Option $OPTARG ignored";;   # DEFAULT
		esac
	done

	if [ "${OPTION_SILENT}" ]; then
		user_sel=`echo user`
	fi

	if [ "${OPTION_CLEANUP}" ]; then
		clean
		exit 0
	fi
}


function check_driver ()
{
	# Copy the news sources and check the driver
	# Syntax:  check_driver
	# Author: mlindner
	# Returns:
	#       N/A

	# Copy driver into the /usr/src/ directory
	echo -n "Copy driver sources into ${KERNEL_SOURCE}"
	cp *.c ${KERNEL_SOURCE}/drivers/net/$drv_name 
	cp h/*.h ${KERNEL_SOURCE}/drivers/net/$drv_name/h/ 
	message_status 1 "done"

	echo -n "Copy driver man page into /usr/share/man/man4/"
	cp sk98lin.4 /usr/share/man/man4/
	gzip /usr/share/man/man4/sk98lin.4
	message_status 1 "done"

	# Check the driver
	fname="Check the driver"
	echo -n $fname
	message_status 3 "working"

	echo "Check the driver" >> $logfile 2>&1
	echo "====================================" >> $logfile 2>&1
	sync
	sleep 1
	sync
	$insmod_bin ./$drv_name.o >> $logfile 2>&1
	insmod_count=`lsmod | grep $drv_name -c`
	error_dev_count=`grep -c "No such device" $logfile`

	if [ $insmod_count != 1 ]; then
		if [ $error_dev_count != 0 ]; then
		echo -en "\015"
		echo -n $fname
		message_status 0 "warning"
		echo
		echo "An error has occurred during the check proces which prevented  "; \
		echo "the installation from completing.                              "; \
		echo "It seems as you don't have pluged any Marvell Yukon or         "; \
		echo "SysKonnect SK-98xx/SK-95xx Gigabit Ethernet Adapter in.        "; \
		echo "Please plug a card into your machine and start the setup again "; \
		echo "or copy the driver manually into the modules directory:        "; \
		echo
		echo "  cp $drv_name.o /lib/modules/${KERNEL_VERSION}/kernel/drivers/net/$drv_name/"; \
		echo

		echo $inst_failed
		cleanup
		clean
		exit 1
		else
		echo -en "\015"
		echo -n $fname
		message_status 0 "error"	
		echo "An error has occurred during the check proces which prevented  "; \
		echo "the installation from completing.                              "; \
		echo "It's not possible to build a standalone $drv_name driver on this "; \
		echo "host. The kernel don't export a neccesary symbols for the      "; \
		echo "device driver and we aren't able to load the driver.           "; \
		echo 
		echo "Please compile the kernel and the driver manually.             "; \
		echo "The new driver has been installed in the /usr/src/linux        "; \
		echo "directory.                                                     "; \
		echo 
		echo "   1.) Go to the directory /usr/src/linux                      "; \
		echo "   2.) For the console mode, run the command: make menuconfig  "; \
		echo "   3.) Select the options you want to compile into the kernel  "; \
		echo "   4.) Select the menu \"Network Device Support\"              "; \
		echo "   5.) Select \"Ethernet (1000 Mbit)\".                        "; \
		echo "   5.) Mark \"Marvell Yukon/SysKonnect SK-98xx/SK-95xx Gigabit "; \
		echo "       Ethernet Adapter support\" with (M)                     "; \
		echo "   6.) Execute the command:                                    "; \
		echo "           make dep clean bzImage modules modules_install      "; \
		echo "   7.) Install the new kernel                                  "; \

		echo "+++ Check error. Insmod error!" >> $logfile 2>&1
		echo $inst_failed
		fi
	else
		echo -en "\015"
		echo -n $fname
		message_status 1 "done"
	fi


	# Copy the driver
	fname="Copying driver"
	echo -n $fname
	message_status 3 "working"
	cp $drv_name.o /lib/modules/${KERNEL_VERSION}/kernel/drivers/net/$drv_name/
	echo -en "\015"
	echo -n $fname
	message_status 1 "done"
}


function version_h_management ()
{
	# Save or restore version.h file
	# Syntax: version_h_management FLAG
	#	FLAG:
	#		1 == Save version.h
	#		1 != Restore version.h
	# Author: mlindner
	# Returns:
	#       N/A

	if [ $1 != 1 ]; then
	# Restore version.h
		if [ "$user_sel" == "user" ]; then
			if [ -e ${TMP_DIR}/version.h ]; then
				echo -n "Restore old version.h"
				cp ${TMP_DIR}/version.h ${KERNEL_HEADER}/linux/version.h
				message_status 1 "done"
			fi
		fi
	else
	# Save old version.h file
		if [ -f ${KERNEL_SOURCE}/include/linux/version.h ]; then
			echo -n "Save old version.h file"
			cp ${KERNEL_SOURCE}/include/linux/version.h ${TMP_DIR}/
			message_status 1 "done"
		fi
	fi

}


function cleanup_trap ()
{
	# Sig handling
	# Syntax: cleanup_trap
	#	FLAG:
	# Author: mlindner
	# Returns:
	#       N/A

	cleanup
	clean
	exit 1
	
}


function autoconf_h_management ()
{
	# Save or restore autoconf.h file
	# Syntax: autoconf_h_management FLAG
	#	FLAG:
	#		1 == Save autoconf.h
	#		1 != Restore autoconf.h
	# Author: mlindner
	# Returns:
	#       N/A

	if [ $1 != 1 ]; then
	# Restore autoconf.h
		if [ "$user_sel" == "user" ]; then
			if [ -e ${TMP_DIR}/autoconf.h ]; then
				echo -n "Restore old autoconf.h"
				cp ${TMP_DIR}/autoconf.h ${KERNEL_HEADER}/linux/autoconf.h
				message_status 1 "done"
			fi
		fi
	else
	# Save old autoconf.h file
		if [ -f ${KERNEL_SOURCE}/include/linux/autoconf.h ]; then
			echo -n "Save old autoconf.h file"
			cp ${KERNEL_SOURCE}/include/linux/autoconf.h ${TMP_DIR}/
			message_status 1 "done"
		fi
	fi

}



# Start
#####################################################################

drv_name=`echo sk98lin`
working_dir=`pwd`
logfile="$working_dir/install.log"
rm -rf $logfile &> /dev/null
trap cleanup_trap INT TERM
KERNEL_TREE_MAKE=0;
VERSION="1.09 (20030923)"

# Extract all given parameters
extract_params $*

# Print copyright informations, mode selection and check
start_sequence

# Generate safe tmp dir
make_safe_tmp_dir

# Check user informations and tools availability
check_user_and_tools

# Check kernel and module informations
check_kernel_informations

# Create tmp dir and unpack the driver
unpack_driver

# Create makefile
create_makefile

# Save old version.h file
version_h_management 1

# Save old autoconf.h file
autoconf_h_management 1

# Check config files
generate_config

# Configure, check and build the driver
make_driver

# Restore version.h
version_h_management 0

# Restore old autoconf.h file
autoconf_h_management 0

# Copy sources and check the new driver (only if silent mode disabled)
if [ ! "${OPTION_SILENT}" ]; then
	check_driver
	cleanup
	clean

	echo "All done. Driver installed and loaded."
	echo "To load the module manually, proceed as follows:"
	echo "      Enter \"modprobe $drv_name\""
	echo 
	echo "                                                     Have fun..."
else
#	if [ ${SK_RPM_BUILD_ROOT} ]; then
#		cp $drv_name.o ${SK_RPM_BUILD_ROOT}
#	fi
	cp $drv_name.o $working_dir
	cleanup
	echo "All done. Driver compiled."
fi

# Cleanup tmp files
exit 0

