/* via926: A NE2000 clone on PCI bus driver for Linux. */
/*
	A Linux device driver for PCI NE2000 clones.

	Authorship and other copyrights:
	1992-1998 by Donald Becker, NE2000 core and various modifications.
	1995-1998 by Paul Gortmaker, core modifications and PCI support.

	Copyright 1993 assigned to the United States Government as represented
	by the Director, National Security Agency.

	This software may be used and distributed according to the terms
	of the GNU Public License, incorporated herein by reference.

	The author may be reached as becker@CESDIS.gsfc.nasa.gov, or C/O
	Center of Excellence in Space Data and Information Sciences
	Code 930.5, Goddard Space Flight Center, Greenbelt MD 20771

	People are making PCI ne2000 clones! Oh the horror, the horror...

	Issues remaining:
	No full-duplex support.
*/

/* Our copyright info must remain in the binary. */
static const char *version =
"via926.c:v1.00 2/1/99 \n";

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/pci.h>
#include <linux/bios32.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include "8390.h"

/* Set statically or when loading the driver module. */
static debug = 1;

/* Some defines that people can play with if so inclined. */

/* Do #define LOAD_8390_BY_KERNELD to automatically load 8390 support. */
#ifdef LOAD_8390_BY_KERNELD
#include <linux/kerneld.h>
#endif
/* Use 32 bit data-movement operations instead of 16 bit. */
/*#define USE_LONGIO
*/
/* Do we implement the read before write bugfix ? */
/* #define NE_RW_BUGFIX */

/* Do we have a non std. amount of memory? (in units of 256 byte pages) */
/* #define PACKETBUF_MEMSIZE	0x40 */

static struct {
	unsigned short vendor, dev_id;
	char *name;
}
pci_clone_list[] = {
	{0x1106, 0x0926, "Via 82C926"},
	{0,}
};

/* ---- No user-serviceable parts below ---- */

#define NE_BASE	 (dev->base_addr)
#define NE_CMD	 	0x00
#define NE_DATAPORT	0x10	/* NatSemi-defined port window offset. */
#define NE_RESET	0x1f	/* Issue a read to reset, a write to clear. */
#define NE_IO_EXTENT	0x20

#define NESM_START_PG	0x40	/* First page of TX buffer */
#define NESM_STOP_PG	0x80	/* Last page +1 of RX ring */

int ne2k_pci_probe(struct device *dev);
static struct device *ne2k_pci_probe1(struct device *dev, int ioaddr, int irq);

static int ne_open(struct device *dev);
static int ne_close(struct device *dev);

static void ne_reset_8390(struct device *dev);
static void ne_get_8390_hdr(struct device *dev, struct e8390_pkt_hdr *hdr,
			  int ring_page);
static void ne_block_input(struct device *dev, int count,
			  struct sk_buff *skb, int ring_offset);
static void ne_block_output(struct device *dev, const int count,
		const unsigned char *buf, const int start_page);



/* No room in the standard 8390 structure for extra info we need. */
struct ne2k_pci_card {
	struct ne2k_pci_card *next;
	struct device *dev;
	unsigned char pci_bus, pci_device_fn;
};
/* A list of all installed devices, for removing the driver module. */
static struct ne2k_pci_card *ne2k_card_list = NULL;

#ifdef LOAD_8390_BY_KERNELD
static int (*Lethdev_init)(struct device *dev);
static void (*LNS8390_init)(struct device *dev, int startp);
static int (*Lei_open)(struct device *dev);
static int (*Lei_close)(struct device *dev);
static void (*Lei_interrupt)(int irq, void *dev_id, struct pt_regs *regs);
#else
#define Lethdev_init ethdev_init
#define LNS8390_init NS8390_init
#define Lei_open ei_open
#define Lei_close ei_close
#define Lei_interrupt ei_interrupt
#endif

#ifdef MODULE

int
init_module(void)
{
	/* We must emit version information. */
	if (debug)
		printk(KERN_INFO "%s", version);

	return ne2k_pci_probe(0);
}

void
cleanup_module(void)
{
	struct device *dev;
	struct ne2k_pci_card *this_card;

	/* No need to check MOD_IN_USE, as sys_delete_module() checks. */
	while (ne2k_card_list) {
		dev = ne2k_card_list->dev;
		unregister_netdev(dev);
		release_region(dev->base_addr, NE_IO_EXTENT);
		kfree(dev);
		this_card = ne2k_card_list;
		ne2k_card_list = ne2k_card_list->next;
		kfree(this_card);
	}

#ifdef LOAD_8390_BY_KERNELD
	release_module("8390", 0);
#endif
}

#endif  /* MODULE */

/*
  NEx000-clone boards have a Station Address (SA) PROM (SAPROM) in the packet
  buffer memory space.  By-the-spec NE2000 clones have 0x57,0x57 in bytes
  0x0e,0x0f of the SAPROM, while other supposed NE2000 clones must be
  detected by their SA prefix.

  Reading the SAPROM from a word-wide card with the 8390 set in byte-wide
  mode results in doubled values, which can be detected and compensated for.

  The probe is also responsible for initializing the card and filling
  in the 'dev' and 'ei_status' structures.
*/

#ifdef HAVE_DEVLIST
struct netdev_entry netcard_drv =
{"ne2k_pci", ne2k_pci_probe1, NE_IO_EXTENT, 0};
#endif

int ne2k_pci_probe(struct device *dev)
{
	static int pci_index = 0;	/* Static, for multiple calls. */
	int cards_found = 0;
	int i;

	if ( ! pcibios_present())
		return -ENODEV;

#ifndef MODULE
	{
		static unsigned version_printed = 0;
		if (version_printed++ == 0)
			printk(KERN_INFO "%s", version);
	}
#endif

	for (;pci_index < 0xff; pci_index++) {
		unsigned char pci_bus, pci_device_fn;
		u8 pci_irq_line;
		u16 pci_command, new_command, vendor, device;
		u32 pci_ioaddr;

		if (pcibios_find_class (PCI_CLASS_NETWORK_ETHERNET << 8, pci_index,
								&pci_bus, &pci_device_fn)
			!= PCIBIOS_SUCCESSFUL)
			break;
		pcibios_read_config_word(pci_bus, pci_device_fn,
								 PCI_VENDOR_ID, &vendor);
		pcibios_read_config_word(pci_bus, pci_device_fn,
								 PCI_DEVICE_ID, &device);

		/* Note: some vendor IDs (RealTek) have non-NE2k cards as well. */
		for (i = 0; pci_clone_list[i].vendor != 0; i++)
			if (pci_clone_list[i].vendor == vendor
				&& pci_clone_list[i].dev_id == device)
				break;
		if (pci_clone_list[i].vendor == 0)
			continue;

		pcibios_read_config_dword(pci_bus, pci_device_fn,
								  PCI_BASE_ADDRESS_0, &pci_ioaddr);
		pcibios_read_config_byte(pci_bus, pci_device_fn,
								 PCI_INTERRUPT_LINE, &pci_irq_line);
		pcibios_read_config_word(pci_bus, pci_device_fn,
								 PCI_COMMAND, &pci_command);

		/* Remove I/O space marker in bit 0. */
		pci_ioaddr &= PCI_BASE_ADDRESS_IO_MASK;

		/* Avoid already found cards from previous calls */
		if (check_region(pci_ioaddr, NE_IO_EXTENT))
			continue;

		/* Activate the card: fix for brain-damaged Win98 BIOSes. */
		new_command = pci_command | PCI_COMMAND_IO;
		if (pci_command != new_command) {
			printk(KERN_INFO "  The PCI BIOS has not enabled this"
				   " NE2k clone!  Updating PCI command %4.4x->%4.4x.\n",
				   pci_command, new_command);
			pcibios_write_config_word(pci_bus, pci_device_fn,
									  PCI_COMMAND, new_command);
		}

		if (pci_irq_line <= 0 || pci_irq_line >= NR_IRQS)
			printk(KERN_WARNING " WARNING: The PCI BIOS assigned this PCI NE2k"
				   " card to IRQ %d, which is unlikely to work!.\n"
				   KERN_WARNING " You should use the PCI BIOS setup to assign"
				   " a valid IRQ line.\n", pci_irq_line);

		printk("via926.c: PCI NE2000 clone '%s' at I/O %#x, IRQ %d.\n",
			   pci_clone_list[i].name, pci_ioaddr, pci_irq_line);
		dev = ne2k_pci_probe1(dev, pci_ioaddr, pci_irq_line);
		if (dev == 0) {
			/* Should not happen. */
			printk(KERN_ERR "via926: Probe of PCI card at %#x failed.\n",
				   pci_ioaddr);
			continue;
		} else {
			struct ne2k_pci_card *ne2k_card =
				kmalloc(sizeof(struct ne2k_pci_card), GFP_KERNEL);
			ne2k_card->next = ne2k_card_list;
			ne2k_card_list = ne2k_card;
			ne2k_card->dev = dev;
			ne2k_card->pci_bus = pci_bus;
			ne2k_card->pci_device_fn = pci_device_fn;
		}
		dev = 0;

		cards_found++;
	}

	return cards_found ? 0 : -ENODEV;
}

static struct device *ne2k_pci_probe1(struct device *dev, int ioaddr, int irq)
{
	int i;
	unsigned char SA_prom[32];
	const char *name = NULL;
	int start_page, stop_page;
	int reg0 = inb(ioaddr);

	if (reg0 == 0xFF)
		return 0;

	/* Do a preliminary verification that we have a 8390. */
	{
		int regd;
		outb(E8390_NODMA+E8390_PAGE1+E8390_STOP, ioaddr + E8390_CMD);
		regd = inb(ioaddr + 0x0d);
		outb(0xff, ioaddr + 0x0d);
		outb(E8390_NODMA+E8390_PAGE0, ioaddr + E8390_CMD);
		inb(ioaddr + EN0_COUNTER0); /* Clear the counter by reading. */
		if (inb(ioaddr + EN0_COUNTER0) != 0) {
			outb(reg0, ioaddr);
			outb(regd, ioaddr + 0x0d);	/* Restore the old values. */
			return 0;
		}
	}

	dev = init_etherdev(dev, 0);

	/* Reset card. Who knows what dain-bramaged state it was left in. */
	{
		unsigned long reset_start_time = jiffies;

		outb(inb(ioaddr + NE_RESET), ioaddr + NE_RESET);

		/* This looks like a horrible timing loop, but it should never take
		   more than a few cycles.
		*/
		while ((inb(ioaddr + EN0_ISR) & ENISR_RESET) == 0)
			/* Limit wait: '2' avoids jiffy roll-over. */
			if (jiffies - reset_start_time > 2) {
				printk("via926: Card failure (no reset ack).\n");
				return 0;
			}
		
		outb(0xff, ioaddr + EN0_ISR);		/* Ack all intr. */
	}

#if defined(LOAD_8390_BY_KERNELD)
	/* We are now certain the 8390 module is required. */
	if (request_module("8390")) {
		printk("via926: Failed to load the 8390 core module.\n");
		return 0;
	}
	if ((Lethdev_init = (void*)get_module_symbol(0, "ethdev_init")) == 0 ||
		(LNS8390_init = (void*)get_module_symbol(0, "NS8390_init")) == 0 ||
		(Lei_open = (void*)get_module_symbol(0, "ei_open")) == 0 ||
		(Lei_close = (void*)get_module_symbol(0, "ei_close")) == 0 ||
		(Lei_interrupt = (void*)get_module_symbol(0, "ei_interrupt")) == 0 ) {
		printk("via926: Failed to resolve an 8390 symbol.\n");
		release_module("8390", 0);
		return 0;
	}
#endif

	/* Read the 16 bytes of station address PROM.
	   We must first initialize registers, similar to NS8390_init(eifdev, 0).
	   We can't reliably read the SAPROM address without this.
	   (I learned the hard way!). */
	{
		struct {unsigned char value, offset; } program_seq[] = {
			{E8390_NODMA+E8390_PAGE0+E8390_STOP, E8390_CMD}, /* Select page 0*/
			{0x49,	EN0_DCFG},	/* Set word-wide access. */
			{0x00,	EN0_RCNTLO},	/* Clear the count regs. */
			{0x00,	EN0_RCNTHI},
			{0x00,	EN0_IMR},	/* Mask completion irq. */
			{0xFF,	EN0_ISR},
			{E8390_RXOFF, EN0_RXCR},	/* 0x20  Set to monitor */
			{E8390_TXOFF, EN0_TXCR},	/* 0x02  and loopback mode. */
			{32,	EN0_RCNTLO},
			{0x00,	EN0_RCNTHI},
			{0x00,	EN0_RSARLO},	/* DMA starting at 0x0000. */
			{0x00,	EN0_RSARHI},
			{E8390_RREAD+E8390_START, E8390_CMD},
		};
		for (i = 0; i < sizeof(program_seq)/sizeof(program_seq[0]); i++)
			outb(program_seq[i].value, ioaddr + program_seq[i].offset);

	}

#ifdef notdef
	/* Some broken PCI cards don't respect the byte-wide
	   request in program_seq above, and hence don't have doubled up values.
	*/
	for(i = 0; i < 32 /*sizeof(SA_prom)*/; i+=2) {
		SA_prom[i] = inb(ioaddr + NE_DATAPORT);
		SA_prom[i+1] = inb(ioaddr + NE_DATAPORT);
		if (SA_prom[i] != SA_prom[i+1])
			sa_prom_doubled = 0;
	}

	if (sa_prom_doubled)
		for (i = 0; i < 16; i++)
			SA_prom[i] = SA_prom[i+i];
#else
	for(i = 0; i < 32 /*sizeof(SA_prom)*/; i++)
		SA_prom[i] = inb(ioaddr + NE_DATAPORT);

#endif

	/* We always set the 8390 registers for word mode. */
	outb(0x49, ioaddr + EN0_DCFG);
	start_page = NESM_START_PG;
	stop_page = NESM_STOP_PG;

	/* Set up the rest of the parameters. */
	name = "PCI NE2000";

	dev->irq = irq;
	dev->base_addr = ioaddr;

	/* Allocate dev->priv and fill in 8390 specific dev fields. */
	if (Lethdev_init(dev)) {
		printk ("%s: unable to get memory for dev->priv.\n", dev->name);
		return 0;
	}

	request_region(ioaddr, NE_IO_EXTENT, dev->name);

	printk("%s: %s found at %#x, IRQ %d, ",
		   dev->name, name, ioaddr, dev->irq);
	for(i = 0; i < 6; i++) {
		printk("%2.2X%s", SA_prom[i], i == 5 ? ".\n": ":");
		dev->dev_addr[i] = SA_prom[i];
	}

	ei_status.name = name;
	ei_status.tx_start_page = start_page;
	ei_status.stop_page = stop_page;
	ei_status.word16 = 1;

	ei_status.rx_start_page = start_page + TX_PAGES;
#ifdef PACKETBUF_MEMSIZE
	/* Allow the packet buffer size to be overridden by know-it-alls. */
	ei_status.stop_page = ei_status.tx_start_page + PACKETBUF_MEMSIZE;
#endif

	ei_status.reset_8390 = &ne_reset_8390;
	ei_status.block_input = &ne_block_input;
	ei_status.block_output = &ne_block_output;
	ei_status.get_8390_hdr = &ne_get_8390_hdr;
	dev->open = &ne_open;
	dev->stop = &ne_close;
	LNS8390_init(dev, 0);
	return dev;
}

static int
ne_open(struct device *dev)
{
	if (request_irq(dev->irq, Lei_interrupt, SA_SHIRQ, dev->name, dev))
		return -EAGAIN;
	Lei_open(dev);
	MOD_INC_USE_COUNT;
	return 0;
}

static int
ne_close(struct device *dev)
{
	Lei_close(dev);
	free_irq(dev->irq, dev);
	MOD_DEC_USE_COUNT;
	return 0;
}

/* Hard reset the card.  This used to pause for the same period that a
   8390 reset command required, but that shouldn't be necessary. */
static void
ne_reset_8390(struct device *dev)
{
	unsigned long reset_start_time = jiffies;

	if (debug > 1) printk("%s: Resetting the 8390 t=%ld...",
						  dev->name, jiffies);

	outb(inb(NE_BASE + NE_RESET), NE_BASE + NE_RESET);

	ei_status.txing = 0;
	ei_status.dmaing = 0;

	/* This check _should_not_ be necessary, omit eventually. */
	while ((inb(NE_BASE+EN0_ISR) & ENISR_RESET) == 0)
		if (jiffies - reset_start_time > 2) {
			printk("%s: ne_reset_8390() did not complete.\n", dev->name);
			break;
		}
	outb(ENISR_RESET, NE_BASE + EN0_ISR);	/* Ack intr. */
}

/* Grab the 8390 specific header. Similar to the block_input routine, but
   we don't need to be concerned with ring wrap as the header will be at
   the start of a page, so we optimize accordingly. */

static void
ne_get_8390_hdr(struct device *dev, struct e8390_pkt_hdr *hdr, int ring_page)
{

	int nic_base = dev->base_addr;

	/* This *shouldn't* happen. If it does, it's the last thing you'll see */
	if (ei_status.dmaing) {
		printk("%s: DMAing conflict in ne_get_8390_hdr "
			   "[DMAstat:%d][irqlock:%d][intr:%d].\n",
			   dev->name, ei_status.dmaing, ei_status.irqlock,
			   dev->interrupt);
		return;
	}

	ei_status.dmaing |= 0x01;
	outb(E8390_NODMA+E8390_PAGE0+E8390_START, nic_base+ NE_CMD);
	outb(sizeof(struct e8390_pkt_hdr), nic_base + EN0_RCNTLO);
	outb(0, nic_base + EN0_RCNTHI);
	outb(0, nic_base + EN0_RSARLO);		/* On page boundary */
	outb(ring_page, nic_base + EN0_RSARHI);
	outb(E8390_RREAD+E8390_START, nic_base + NE_CMD);

#if defined(USE_LONGIO)
	*(u32*)hdr = inl(NE_BASE + NE_DATAPORT);
#else
	insw(NE_BASE + NE_DATAPORT, hdr, sizeof(struct e8390_pkt_hdr)>>1);
#endif

	outb(ENISR_RDC, nic_base + EN0_ISR);	/* Ack intr. */
	ei_status.dmaing &= ~0x01;
}

/* Block input and output, similar to the Crynwr packet driver.  If you
   are porting to a new ethercard, look at the packet driver source for hints.
   The NEx000 doesn't share the on-board packet memory -- you have to put
   the packet out through the "remote DMA" dataport using outb. */

static void
ne_block_input(struct device *dev, int count, struct sk_buff *skb, int ring_offset)
{
	int nic_base = dev->base_addr;
	char *buf = skb->data;

	/* This *shouldn't* happen. If it does, it's the last thing you'll see */
	if (ei_status.dmaing) {
		printk("%s: DMAing conflict in ne_block_input "
			   "[DMAstat:%d][irqlock:%d][intr:%d].\n",
			   dev->name, ei_status.dmaing, ei_status.irqlock,
			   dev->interrupt);
		return;
	}
	ei_status.dmaing |= 0x01;
	outb(E8390_NODMA+E8390_PAGE0+E8390_START, nic_base+ NE_CMD);
	outb(count & 0xff, nic_base + EN0_RCNTLO);
	outb(count >> 8, nic_base + EN0_RCNTHI);
	outb(ring_offset & 0xff, nic_base + EN0_RSARLO);
	outb(ring_offset >> 8, nic_base + EN0_RSARHI);
	outb(E8390_RREAD+E8390_START, nic_base + NE_CMD);

#if defined(USE_LONGIO)
	insl(NE_BASE + NE_DATAPORT, buf, count>>2);
	if (count & 3) {
		buf += count & ~3;
		if (count & 2)
			*((u16*)buf)++ = inw(NE_BASE + NE_DATAPORT);
		if (count & 1)
			*buf = inb(NE_BASE + NE_DATAPORT);
	}
#else
	insw(NE_BASE + NE_DATAPORT,buf,count>>1);
	if (count & 0x01) {
		buf[count-1] = inb(NE_BASE + NE_DATAPORT);
	}
#endif

	outb(ENISR_RDC, nic_base + EN0_ISR);	/* Ack intr. */
	ei_status.dmaing &= ~0x01;
}

static void
ne_block_output(struct device *dev, int count,
		const unsigned char *buf, const int start_page)
{
	int nic_base = NE_BASE;
	unsigned long dma_start;

	/* On little-endian it's always safe to round the count up for
	   word writes. */
	if (count & 0x01)
		count++;

	/* This *shouldn't* happen. If it does, it's the last thing you'll see */
	if (ei_status.dmaing) {
		printk("%s: DMAing conflict in ne_block_output."
			   "[DMAstat:%d][irqlock:%d][intr:%d]\n",
			   dev->name, ei_status.dmaing, ei_status.irqlock,
			   dev->interrupt);
		return;
	}
	ei_status.dmaing |= 0x01;
	/* We should already be in page 0, but to be safe... */
	outb(E8390_PAGE0+E8390_START+E8390_NODMA, nic_base + NE_CMD);

#ifdef NE8390_RW_BUGFIX
	/* Handle the read-before-write bug the same way as the
	   Crynwr packet driver -- the NatSemi method doesn't work.
	   Actually this doesn't always work either, but if you have
	   problems with your NEx000 this is better than nothing! */
	outb(0x42, nic_base + EN0_RCNTLO);
	outb(0x00, nic_base + EN0_RCNTHI);
	outb(0x42, nic_base + EN0_RSARLO);
	outb(0x00, nic_base + EN0_RSARHI);
	outb(E8390_RREAD+E8390_START, nic_base + NE_CMD);
#endif
	outb(ENISR_RDC, nic_base + EN0_ISR);

   /* Now the normal output. */
	outb(count & 0xff, nic_base + EN0_RCNTLO);
	outb(count >> 8,   nic_base + EN0_RCNTHI);
	outb(0x00, nic_base + EN0_RSARLO);
	outb(start_page, nic_base + EN0_RSARHI);
	outb(E8390_RWRITE+E8390_START, nic_base + NE_CMD);
#if defined(USE_LONGIO)
	outsl(NE_BASE + NE_DATAPORT, buf, count>>2);
	if (count & 3) {
		buf += count & ~3;
		if (count & 2)
			outw(*((u16*)buf)++, NE_BASE + NE_DATAPORT);
	}
#else
	outsw(NE_BASE + NE_DATAPORT, buf, count>>1);
#endif

	dma_start = jiffies;

	while ((inb(nic_base + EN0_ISR) & ENISR_RDC) == 0)
		if (jiffies - dma_start > 2) { 			/* Avoid clock roll-over. */
			printk("%s: timeout waiting for Tx RDC.\n", dev->name);
			ne_reset_8390(dev);
			LNS8390_init(dev,1);
			break;
		}

	outb(ENISR_RDC, nic_base + EN0_ISR);	/* Ack intr. */
	ei_status.dmaing &= ~0x01;
	return;
}


/*
 * Local variables:
 *  compile-command: "gcc -DMODVERSIONS  -DMODULE -D__KERNEL__ -Wall -Wstrict-prototypes -O6 -fomit-frame-pointer -I/usr/src/linux/drivers/net/ -c via926.c"
 *  alt-compile-command: "gcc -DMODULE -D__KERNEL__ -Wall -Wstrict-prototypes -O6 -fomit-frame-pointer -I/usr/src/linux/drivers/net/ -c via926.c"
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  version-control: t
 *  kept-new-versions: 5
 * End:
 */
/* 8390.c: A general NS8390 ethernet driver core for linux. */
/*
	Written 1992-94 by Donald Becker.
  
	Copyright 1993 United States Government as represented by the
	Director, National Security Agency.

	This software may be used and distributed according to the terms
	of the GNU Public License, incorporated herein by reference.

	The author may be reached as becker@CESDIS.gsfc.nasa.gov, or C/O
	Center of Excellence in Space Data and Information Sciences
	   Code 930.5, Goddard Space Flight Center, Greenbelt MD 20771
  
  This is the chip-specific code for many 8390-based ethernet adaptors.
  This is not a complete driver, it must be combined with board-specific
  code such as ne.c, wd.c, 3c503.c, etc.

  Seeing how at least eight drivers use this code, (not counting the
  PCMCIA ones either) it is easy to break some card by what seems like
  a simple innocent change. Please contact me or Donald if you think
  you have found something that needs changing. -- PG


  Changelog:

  Paul Gortmaker	: remove set_bit lock, other cleanups.
  Paul Gortmaker	: add ei_get_8390_hdr() so we can pass skb's to 
			  ei_block_input() for eth_io_copy_and_sum().
  Paul Gortmaker	: exchange static int ei_pingpong for a #define,
			  also add better Tx error handling.
  Paul Gortmaker	: rewrite Rx overrun handling as per NS specs.


  Sources:
  The National Semiconductor LAN Databook, and the 3Com 3c503 databook.

  */
/*
static const char *version =
    "8390.c:v1.10 9/23/94 Donald Becker (becker@cesdis.gsfc.nasa.gov)\n";
*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/string.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/in.h>
#include <linux/interrupt.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>

#include "8390.h"

/* These are the operational function interfaces to board-specific
   routines.
	void reset_8390(struct device *dev)
		Resets the board associated with DEV, including a hardware reset of
		the 8390.  This is only called when there is a transmit timeout, and
		it is always followed by 8390_init().
	void block_output(struct device *dev, int count, const unsigned char *buf,
					  int start_page)
		Write the COUNT bytes of BUF to the packet buffer at START_PAGE.  The
		"page" value uses the 8390's 256-byte pages.
	void get_8390_hdr(struct device *dev, struct e8390_hdr *hdr, int ring_page)
		Read the 4 byte, page aligned 8390 header. *If* there is a
		subsequent read, it will be of the rest of the packet.
	void block_input(struct device *dev, int count, struct sk_buff *skb, int ring_offset)
		Read COUNT bytes from the packet buffer into the skb data area. Start 
		reading from RING_OFFSET, the address as the 8390 sees it.  This will always
		follow the read of the 8390 header. 
*/
#define ei_reset_8390 (ei_local->reset_8390)
#define ei_block_output (ei_local->block_output)
#define ei_block_input (ei_local->block_input)
#define ei_get_8390_hdr (ei_local->get_8390_hdr)

/* use 0 for production, 1 for verification, >2 for debug */
#ifdef EI_DEBUG
int ei_debug = EI_DEBUG;
#else
int ei_debug = 1;
#endif

/* Index to functions. */
static void ei_tx_intr(struct device *dev);
static void ei_tx_err(struct device *dev);
static void ei_receive(struct device *dev);
static void ei_rx_overrun(struct device *dev);

/* Routines generic to NS8390-based boards. */
static void NS8390_trigger_send(struct device *dev, unsigned int length,
								int start_page);
static void set_multicast_list(struct device *dev);


/* Open/initialize the board.  This routine goes all-out, setting everything
   up anew at each open, even though many of these registers should only
   need to be set once at boot.
   */
int ei_open(struct device *dev)
{
    struct ei_device *ei_local = (struct ei_device *) dev->priv;

    /* This can't happen unless somebody forgot to call ethdev_init(). */
    if (ei_local == NULL) {
	printk(KERN_EMERG "%s: ei_open passed a non-existent device!\n", dev->name);
	return -ENXIO;
    }
    
    irq2dev_map[dev->irq] = dev;
    NS8390_init(dev, 1);
    dev->start = 1;
    ei_local->irqlock = 0;
    return 0;
}

/* Opposite of above. Only used when "ifconfig <devname> down" is done. */
int ei_close(struct device *dev)
{
    NS8390_init(dev, 0);
    dev->start = 0;
    return 0;
}

static int ei_start_xmit(struct sk_buff *skb, struct device *dev)
{
    int e8390_base = dev->base_addr;
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    int length, send_length, output_page;

/*
 *  We normally shouldn't be called if dev->tbusy is set, but the
 *  existing code does anyway. If it has been too long since the
 *  last Tx, we assume the board has died and kick it.
 */
 
    if (dev->tbusy) {	/* Do timeouts, just like the 8003 driver. */
		int txsr = inb(e8390_base+EN0_TSR), isr;
		int tickssofar = jiffies - dev->trans_start;
		if (tickssofar < TX_TIMEOUT ||	(tickssofar < (TX_TIMEOUT+5) && ! (txsr & ENTSR_PTX))) {
			return 1;
		}
		isr = inb(e8390_base+EN0_ISR);
		if (dev->start == 0) {
			printk("%s: xmit on stopped card\n", dev->name);
			return 1;
		}

		/*
		 * Note that if the Tx posted a TX_ERR interrupt, then the
		 * error will have been handled from the interrupt handler.
		 * and not here.
		 */

		printk(KERN_DEBUG "%s: Tx timed out, %s TSR=%#2x, ISR=%#2x, t=%d.\n",
		   dev->name, (txsr & ENTSR_ABT) ? "excess collisions." :
		   (isr) ? "lost interrupt?" : "cable problem?", txsr, isr, tickssofar);

		if (!isr && !ei_local->stat.tx_packets) {
		   /* The 8390 probably hasn't gotten on the cable yet. */
		   ei_local->interface_num ^= 1;   /* Try a different xcvr.  */
		}

		/* Try to restart the card.  Perhaps the user has fixed something. */
		ei_reset_8390(dev);
		NS8390_init(dev, 1);
		dev->trans_start = jiffies;
    }
    
    /* Sending a NULL skb means some higher layer thinks we've missed an
       tx-done interrupt. Caution: dev_tint() handles the cli()/sti()
       itself. */
    if (skb == NULL) {
		dev_tint(dev);
		return 0;
    }
    
    length = skb->len;
    if (skb->len <= 0)
		return 0;

    /* Mask interrupts from the ethercard. */
    outb_p(0x00, e8390_base + EN0_IMR);
    if (dev->interrupt) {
	printk("%s: Tx request while isr active.\n",dev->name);
	outb_p(ENISR_ALL, e8390_base + EN0_IMR);
	return 1;
    }
    ei_local->irqlock = 1;

    send_length = ETH_ZLEN < length ? length : ETH_ZLEN;

#ifdef EI_PINGPONG

    /*
     * We have two Tx slots available for use. Find the first free
     * slot, and then perform some sanity checks. With two Tx bufs,
     * you get very close to transmitting back-to-back packets. With
     * only one Tx buf, the transmitter sits idle while you reload the
     * card, leaving a substantial gap between each transmitted packet.
     */

    if (ei_local->tx1 == 0) {
	output_page = ei_local->tx_start_page;
	ei_local->tx1 = send_length;
	if (ei_debug  &&  ei_local->tx2 > 0)
		printk("%s: idle transmitter tx2=%d, lasttx=%d, txing=%d.\n",
			dev->name, ei_local->tx2, ei_local->lasttx, ei_local->txing);
    } else if (ei_local->tx2 == 0) {
	output_page = ei_local->tx_start_page + TX_1X_PAGES;
	ei_local->tx2 = send_length;
	if (ei_debug  &&  ei_local->tx1 > 0)
		printk("%s: idle transmitter, tx1=%d, lasttx=%d, txing=%d.\n",
			dev->name, ei_local->tx1, ei_local->lasttx, ei_local->txing);
    } else {	/* We should never get here. */
	if (ei_debug)
		printk("%s: No Tx buffers free! irq=%d tx1=%d tx2=%d last=%d\n",
			dev->name, dev->interrupt, ei_local->tx1, ei_local->tx2, ei_local->lasttx);
	ei_local->irqlock = 0;
	dev->tbusy = 1;
	outb_p(ENISR_ALL, e8390_base + EN0_IMR);
	return 1;
    }

    /*
     * Okay, now upload the packet and trigger a send if the transmitter
     * isn't already sending. If it is busy, the interrupt handler will
     * trigger the send later, upon receiving a Tx done interrupt.
     */

    ei_block_output(dev, length, skb->data, output_page);
    if (! ei_local->txing) {
	ei_local->txing = 1;
	NS8390_trigger_send(dev, send_length, output_page);
	dev->trans_start = jiffies;
	if (output_page == ei_local->tx_start_page) {
		ei_local->tx1 = -1;
		ei_local->lasttx = -1;
	} else {
		ei_local->tx2 = -1;
		ei_local->lasttx = -2;
	}
    } else
	ei_local->txqueue++;

    dev->tbusy = (ei_local->tx1  &&  ei_local->tx2);

#else	/* EI_PINGPONG */

    /*
     * Only one Tx buffer in use. You need two Tx bufs to come close to
     * back-to-back transmits. Expect a 20 -> 25% performance hit on
     * reasonable hardware if you only use one Tx buffer.
     */

    ei_block_output(dev, length, skb->data, ei_local->tx_start_page);
    ei_local->txing = 1;
    NS8390_trigger_send(dev, send_length, ei_local->tx_start_page);
    dev->trans_start = jiffies;
    dev->tbusy = 1;

#endif	/* EI_PINGPONG */

    /* Turn 8390 interrupts back on. */
    ei_local->irqlock = 0;
    outb_p(ENISR_ALL, e8390_base + EN0_IMR);

    dev_kfree_skb (skb, FREE_WRITE);
    
    return 0;
}

/* The typical workload of the driver:
   Handle the ether interface interrupts. */
void ei_interrupt(int irq, void *dev_id, struct pt_regs * regs)
{
    struct device *dev = (struct device *)(irq2dev_map[irq]);
    int e8390_base;
    int interrupts, nr_serviced = 0;
    struct ei_device *ei_local;
    
    if (dev == NULL) {
		printk ("net_interrupt(): irq %d for unknown device.\n", irq);
		return;
    }
    e8390_base = dev->base_addr;
    ei_local = (struct ei_device *) dev->priv;
    if (dev->interrupt || ei_local->irqlock) {
		/* The "irqlock" check is only for testing. */
		printk(ei_local->irqlock
			   ? "%s: Interrupted while interrupts are masked! isr=%#2x imr=%#2x.\n"
			   : "%s: Reentering the interrupt handler! isr=%#2x imr=%#2x.\n",
			   dev->name, inb_p(e8390_base + EN0_ISR),
			   inb_p(e8390_base + EN0_IMR));
		return;
    }
    
    dev->interrupt = 1;
    
    /* Change to page 0 and read the intr status reg. */
    outb_p(E8390_NODMA+E8390_PAGE0, e8390_base + E8390_CMD);
    if (ei_debug > 3)
		printk("%s: interrupt(isr=%#2.2x).\n", dev->name,
			   inb_p(e8390_base + EN0_ISR));
    
    /* !!Assumption!! -- we stay in page 0.	 Don't break this. */
    while ((interrupts = inb_p(e8390_base + EN0_ISR)) != 0
		   && ++nr_serviced < MAX_SERVICE) {
		if (dev->start == 0) {
			printk("%s: interrupt from stopped card\n", dev->name);
			interrupts = 0;
			break;
		}
		if (interrupts & ENISR_OVER) {
			ei_rx_overrun(dev);
		} else if (interrupts & (ENISR_RX+ENISR_RX_ERR)) {
			/* Got a good (?) packet. */
			ei_receive(dev);
		}
		/* Push the next to-transmit packet through. */
		if (interrupts & ENISR_TX) {
			ei_tx_intr(dev);
		} else if (interrupts & ENISR_TX_ERR) {
			ei_tx_err(dev);
		}

		if (interrupts & ENISR_COUNTERS) {
			ei_local->stat.rx_frame_errors += inb_p(e8390_base + EN0_COUNTER0);
			ei_local->stat.rx_crc_errors   += inb_p(e8390_base + EN0_COUNTER1);
			ei_local->stat.rx_missed_errors+= inb_p(e8390_base + EN0_COUNTER2);
			outb_p(ENISR_COUNTERS, e8390_base + EN0_ISR); /* Ack intr. */
		}
		
		/* Ignore any RDC interrupts that make it back to here. */
		if (interrupts & ENISR_RDC) {
			outb_p(ENISR_RDC, e8390_base + EN0_ISR);
		}

		outb_p(E8390_NODMA+E8390_PAGE0+E8390_START, e8390_base + E8390_CMD);
    }
    
    if (interrupts && ei_debug) {
		outb_p(E8390_NODMA+E8390_PAGE0+E8390_START, e8390_base + E8390_CMD);
		if (nr_serviced >= MAX_SERVICE) {
			printk("%s: Too much work at interrupt, status %#2.2x\n",
				   dev->name, interrupts);
			outb_p(ENISR_ALL, e8390_base + EN0_ISR); /* Ack. most intrs. */
		} else {
			printk("%s: unknown interrupt %#2x\n", dev->name, interrupts);
			outb_p(0xff, e8390_base + EN0_ISR); /* Ack. all intrs. */
		}
    }
    dev->interrupt = 0;
    return;
}

/*
 * A transmitter error has happened. Most likely excess collisions (which
 * is a fairly normal condition). If the error is one where the Tx will
 * have been aborted, we try and send another one right away, instead of
 * letting the failed packet sit and collect dust in the Tx buffer. This
 * is a much better solution as it avoids kernel based Tx timeouts, and
 * an unnecessary card reset.
 */

static void ei_tx_err(struct device *dev)
{
    int e8390_base = dev->base_addr;
    unsigned char txsr = inb_p(e8390_base+EN0_TSR);
    unsigned char tx_was_aborted = txsr & (ENTSR_ABT+ENTSR_FU);
    struct ei_device *ei_local = (struct ei_device *) dev->priv;

#ifdef VERBOSE_ERROR_DUMP
    printk(KERN_DEBUG "%s: transmitter error (%#2x): ", dev->name, txsr);
    if (txsr & ENTSR_ABT)
		printk("excess-collisions ");
    if (txsr & ENTSR_ND)
		printk("non-deferral ");
    if (txsr & ENTSR_CRS)
		printk("lost-carrier ");
    if (txsr & ENTSR_FU)
		printk("FIFO-underrun ");
    if (txsr & ENTSR_CDH)
		printk("lost-heartbeat ");
    printk("\n");
#endif

    outb_p(ENISR_TX_ERR, e8390_base + EN0_ISR); /* Ack intr. */

    if (tx_was_aborted)
		ei_tx_intr(dev);

    /*
     * Note: NCR reads zero on 16 collisions so we add them
     * in by hand. Somebody might care...
     */
    if (txsr & ENTSR_ABT)
	ei_local->stat.collisions += 16;
	
}

/* We have finished a transmit: check for errors and then trigger the next
   packet to be sent. */
static void ei_tx_intr(struct device *dev)
{
    int e8390_base = dev->base_addr;
    int status = inb(e8390_base + EN0_TSR);
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    
    outb_p(ENISR_TX, e8390_base + EN0_ISR); /* Ack intr. */

#ifdef EI_PINGPONG

    /*
     * There are two Tx buffers, see which one finished, and trigger
     * the send of another one if it exists.
     */
    ei_local->txqueue--;
    if (ei_local->tx1 < 0) {
	if (ei_local->lasttx != 1 && ei_local->lasttx != -1)
		printk("%s: bogus last_tx_buffer %d, tx1=%d.\n",
			   ei_local->name, ei_local->lasttx, ei_local->tx1);
	ei_local->tx1 = 0;
	dev->tbusy = 0;
	if (ei_local->tx2 > 0) {
		ei_local->txing = 1;
		NS8390_trigger_send(dev, ei_local->tx2, ei_local->tx_start_page + 6);
		dev->trans_start = jiffies;
		ei_local->tx2 = -1,
		ei_local->lasttx = 2;
	} else
		ei_local->lasttx = 20, ei_local->txing = 0;
    } else if (ei_local->tx2 < 0) {
	if (ei_local->lasttx != 2  &&  ei_local->lasttx != -2)
		printk("%s: bogus last_tx_buffer %d, tx2=%d.\n",
			   ei_local->name, ei_local->lasttx, ei_local->tx2);
	ei_local->tx2 = 0;
	dev->tbusy = 0;
	if (ei_local->tx1 > 0) {
		ei_local->txing = 1;
		NS8390_trigger_send(dev, ei_local->tx1, ei_local->tx_start_page);
		dev->trans_start = jiffies;
		ei_local->tx1 = -1;
		ei_local->lasttx = 1;
	} else
		ei_local->lasttx = 10, ei_local->txing = 0;
    } else
	printk("%s: unexpected TX-done interrupt, lasttx=%d.\n",
		   dev->name, ei_local->lasttx);

#else	/* EI_PINGPONG */
    /*
     *  Single Tx buffer: mark it free so another packet can be loaded.
     */
    ei_local->txing = 0;
    dev->tbusy = 0;
#endif

    /* Minimize Tx latency: update the statistics after we restart TXing. */
    if (status & ENTSR_COL)
	ei_local->stat.collisions++;
    if (status & ENTSR_PTX)
	ei_local->stat.tx_packets++;
    else {
	ei_local->stat.tx_errors++;
	if (status & ENTSR_ABT) ei_local->stat.tx_aborted_errors++;
	if (status & ENTSR_CRS) ei_local->stat.tx_carrier_errors++;
	if (status & ENTSR_FU)  ei_local->stat.tx_fifo_errors++;
	if (status & ENTSR_CDH) ei_local->stat.tx_heartbeat_errors++;
	if (status & ENTSR_OWC) ei_local->stat.tx_window_errors++;
    }

    mark_bh (NET_BH);
}

/* We have a good packet(s), get it/them out of the buffers. */

static void ei_receive(struct device *dev)
{
    int e8390_base = dev->base_addr;
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    unsigned char rxing_page, this_frame, next_frame;
    unsigned short current_offset;
    int rx_pkt_count = 0;
    struct e8390_pkt_hdr rx_frame;
    int num_rx_pages = ei_local->stop_page-ei_local->rx_start_page;
    
    while (++rx_pkt_count < 10) {
		int pkt_len;
		
		/* Get the rx page (incoming packet pointer). */
		outb_p(E8390_NODMA+E8390_PAGE1, e8390_base + E8390_CMD);
		rxing_page = inb_p(e8390_base + EN1_CURPAG);
		outb_p(E8390_NODMA+E8390_PAGE0, e8390_base + E8390_CMD);
		
		/* Remove one frame from the ring.  Boundary is always a page behind. */
		this_frame = inb_p(e8390_base + EN0_BOUNDARY) + 1;
		if (this_frame >= ei_local->stop_page)
			this_frame = ei_local->rx_start_page;
		
		/* Someday we'll omit the previous, iff we never get this message.
		   (There is at least one clone claimed to have a problem.)  */
		if (ei_debug > 0  &&  this_frame != ei_local->current_page)
			printk("%s: mismatched read page pointers %2x vs %2x.\n",
				   dev->name, this_frame, ei_local->current_page);
		
		if (this_frame == rxing_page)	/* Read all the frames? */
			break;				/* Done for now */
		
		current_offset = this_frame << 8;
		ei_get_8390_hdr(dev, &rx_frame, this_frame);
		
		pkt_len = rx_frame.count - sizeof(struct e8390_pkt_hdr);
		
		next_frame = this_frame + 1 + ((pkt_len+4)>>8);
		
		/* Check for bogosity warned by 3c503 book: the status byte is never
		   written.  This happened a lot during testing! This code should be
		   cleaned up someday. */
		if (rx_frame.next != next_frame
			&& rx_frame.next != next_frame + 1
			&& rx_frame.next != next_frame - num_rx_pages
			&& rx_frame.next != next_frame + 1 - num_rx_pages) {
			ei_local->current_page = rxing_page;
			outb(ei_local->current_page-1, e8390_base+EN0_BOUNDARY);
			ei_local->stat.rx_errors++;
			continue;
		}

		if (pkt_len < 60  ||  pkt_len > 1518) {
			if (ei_debug)
				printk("%s: bogus packet size: %d, status=%#2x nxpg=%#2x.\n",
					   dev->name, rx_frame.count, rx_frame.status,
					   rx_frame.next);
			ei_local->stat.rx_errors++;
		} else if ((rx_frame.status & 0x0F) == ENRSR_RXOK) {
			struct sk_buff *skb;
			
			skb = dev_alloc_skb(pkt_len+2);
			if (skb == NULL) {
				if (ei_debug > 1)
					printk("%s: Couldn't allocate a sk_buff of size %d.\n",
						   dev->name, pkt_len);
				ei_local->stat.rx_dropped++;
				break;
			} else {
				skb_reserve(skb,2);	/* IP headers on 16 byte boundaries */
				skb->dev = dev;
				skb_put(skb, pkt_len);	/* Make room */
				ei_block_input(dev, pkt_len, skb, current_offset + sizeof(rx_frame));
				skb->protocol=eth_type_trans(skb,dev);
				netif_rx(skb);
				ei_local->stat.rx_packets++;
			}
		} else {
			int errs = rx_frame.status;
			if (ei_debug)
				printk("%s: bogus packet: status=%#2x nxpg=%#2x size=%d\n",
					   dev->name, rx_frame.status, rx_frame.next,
					   rx_frame.count);
			if (errs & ENRSR_FO)
				ei_local->stat.rx_fifo_errors++;
		}
		next_frame = rx_frame.next;
		
		/* This _should_ never happen: it's here for avoiding bad clones. */
		if (next_frame >= ei_local->stop_page) {
			printk("%s: next frame inconsistency, %#2x\n", dev->name,
				   next_frame);
			next_frame = ei_local->rx_start_page;
		}
		ei_local->current_page = next_frame;
		outb_p(next_frame-1, e8390_base+EN0_BOUNDARY);
    }

    /* We used to also ack ENISR_OVER here, but that would sometimes mask
    a real overrun, leaving the 8390 in a stopped state with rec'vr off. */
    outb_p(ENISR_RX+ENISR_RX_ERR, e8390_base+EN0_ISR);
    return;
}

/* 
 * We have a receiver overrun: we have to kick the 8390 to get it started
 * again. Problem is that you have to kick it exactly as NS prescribes in
 * the updated datasheets, or "the NIC may act in an unpredictable manner."
 * This includes causing "the NIC to defer indefinitely when it is stopped
 * on a busy network."  Ugh.
 */
static void ei_rx_overrun(struct device *dev)
{
    int e8390_base = dev->base_addr;
    unsigned long wait_start_time;
    unsigned char was_txing, must_resend = 0;
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    
    /*
     * Record whether a Tx was in progress and then issue the
     * stop command.
     */
    was_txing = inb_p(e8390_base+E8390_CMD) & E8390_TRANS;
    outb_p(E8390_NODMA+E8390_PAGE0+E8390_STOP, e8390_base+E8390_CMD);
    
    if (ei_debug > 1)
	printk("%s: Receiver overrun.\n", dev->name);
    ei_local->stat.rx_over_errors++;
    
    /* 
     * Wait a full Tx time (1.2ms) + some guard time, NS says 1.6ms total.
     * Early datasheets said to poll the reset bit, but now they say that
     * it "is not a reliable indicator and subsequently should be ignored."
     * We wait at least 10ms.
     */
    wait_start_time = jiffies;
    while (jiffies - wait_start_time <= 1*HZ/100)
	barrier();

    /*
     * Reset RBCR[01] back to zero as per magic incantation.
     */
    outb_p(0x00, e8390_base+EN0_RCNTLO);
    outb_p(0x00, e8390_base+EN0_RCNTHI);

    /*
     * See if any Tx was interrupted or not. According to NS, this
     * step is vital, and skipping it will cause no end of havoc.
     */
    if (was_txing) { 
	unsigned char tx_completed = inb_p(e8390_base+EN0_ISR) & (ENISR_TX+ENISR_TX_ERR);
	if (!tx_completed) must_resend = 1;
    }

    /*
     * Have to enter loopback mode and then restart the NIC before
     * you are allowed to slurp packets up off the ring.
     */
    outb_p(E8390_TXOFF, e8390_base + EN0_TXCR);
    outb_p(E8390_NODMA + E8390_PAGE0 + E8390_START, e8390_base + E8390_CMD);

    /*
     * Clear the Rx ring of all the debris, and ack the interrupt.
     */
    ei_receive(dev);
    outb_p(ENISR_OVER, e8390_base+EN0_ISR);

    /*
     * Leave loopback mode, and resend any packet that got stopped.
     */
    outb_p(E8390_TXCONFIG, e8390_base + EN0_TXCR); 
    if (must_resend)
    	outb_p(E8390_NODMA + E8390_PAGE0 + E8390_START + E8390_TRANS, e8390_base + E8390_CMD);
	
}

static struct enet_statistics *get_stats(struct device *dev)
{
    short ioaddr = dev->base_addr;
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    
    /* If the card is stopped, just return the present stats. */
    if (dev->start == 0) return &ei_local->stat;

    /* Read the counter registers, assuming we are in page 0. */
    ei_local->stat.rx_frame_errors += inb_p(ioaddr + EN0_COUNTER0);
    ei_local->stat.rx_crc_errors   += inb_p(ioaddr + EN0_COUNTER1);
    ei_local->stat.rx_missed_errors+= inb_p(ioaddr + EN0_COUNTER2);
    
    return &ei_local->stat;
}

/*
 *	Set or clear the multicast filter for this adaptor.
 */
 
static void set_multicast_list(struct device *dev)
{
	short ioaddr = dev->base_addr;
    
	if(dev->flags&IFF_PROMISC)
	{
		outb_p(E8390_RXCONFIG | 0x18, ioaddr + EN0_RXCR);
	}
	else if((dev->flags&IFF_ALLMULTI)||dev->mc_list)
	{
		/* The multicast-accept list is initialized to accept-all, and we
		   rely on higher-level filtering for now. */
		outb_p(E8390_RXCONFIG | 0x08, ioaddr + EN0_RXCR);
	} 
	else
		outb_p(E8390_RXCONFIG, ioaddr + EN0_RXCR);
}

/* Initialize the rest of the 8390 device structure. */
int ethdev_init(struct device *dev)
{
    if (ei_debug > 1)
		printk(version);
    
    if (dev->priv == NULL) {
		struct ei_device *ei_local;
		
		dev->priv = kmalloc(sizeof(struct ei_device), GFP_KERNEL);
		if (dev->priv == NULL)
			return -ENOMEM;
		memset(dev->priv, 0, sizeof(struct ei_device));
		ei_local = (struct ei_device *)dev->priv;
    }
    
    dev->hard_start_xmit = &ei_start_xmit;
    dev->get_stats	= get_stats;
    dev->set_multicast_list = &set_multicast_list;

    ether_setup(dev);
        
    return 0;
}


/* This page of functions should be 8390 generic */
/* Follow National Semi's recommendations for initializing the "NIC". */
void NS8390_init(struct device *dev, int startp)
{
    int e8390_base = dev->base_addr;
    struct ei_device *ei_local = (struct ei_device *) dev->priv;
    int i;
    int endcfg = ei_local->word16 ? (0x48 | ENDCFG_WTS) : 0x48;
    unsigned long flags;
    
    /* Follow National Semi's recommendations for initing the DP83902. */
    outb_p(E8390_NODMA+E8390_PAGE0+E8390_STOP, e8390_base); /* 0x21 */
    outb_p(endcfg, e8390_base + EN0_DCFG);	/* 0x48 or 0x49 */
    /* Clear the remote byte count registers. */
    outb_p(0x00,  e8390_base + EN0_RCNTLO);
    outb_p(0x00,  e8390_base + EN0_RCNTHI);
    /* Set to monitor and loopback mode -- this is vital!. */
    outb_p(E8390_RXOFF, e8390_base + EN0_RXCR); /* 0x20 */
    outb_p(E8390_TXOFF, e8390_base + EN0_TXCR); /* 0x02 */
    /* Set the transmit page and receive ring. */
    outb_p(ei_local->tx_start_page,	 e8390_base + EN0_TPSR);
    ei_local->tx1 = ei_local->tx2 = 0;
    outb_p(ei_local->rx_start_page,	 e8390_base + EN0_STARTPG);
    outb_p(ei_local->stop_page-1, e8390_base + EN0_BOUNDARY); /* 3c503 says 0x3f,NS0x26*/
    ei_local->current_page = ei_local->rx_start_page;		/* assert boundary+1 */
    outb_p(ei_local->stop_page,	  e8390_base + EN0_STOPPG);
    /* Clear the pending interrupts and mask. */
    outb_p(0xFF, e8390_base + EN0_ISR);
    outb_p(0x00,  e8390_base + EN0_IMR);
    
    /* Copy the station address into the DS8390 registers,
       and set the multicast hash bitmap to receive all multicasts. */
    save_flags(flags);
    cli();
    outb_p(E8390_NODMA + E8390_PAGE1 + E8390_STOP, e8390_base); /* 0x61 */
    for(i = 0; i < 6; i++) {
		outb_p(dev->dev_addr[i], e8390_base + EN1_PHYS + i);
    }
    /* Initialize the multicast list to accept-all.  If we enable multicast
       the higher levels can do the filtering. */
    for(i = 0; i < 8; i++)
		outb_p(0xff, e8390_base + EN1_MULT + i);
    
    outb_p(ei_local->rx_start_page,	 e8390_base + EN1_CURPAG);
    outb_p(E8390_NODMA+E8390_PAGE0+E8390_STOP, e8390_base);
    restore_flags(flags);
    dev->tbusy = 0;
    dev->interrupt = 0;
    ei_local->tx1 = ei_local->tx2 = 0;
    ei_local->txing = 0;
    if (startp) {
		outb_p(0xff,  e8390_base + EN0_ISR);
		outb_p(ENISR_ALL,  e8390_base + EN0_IMR);
		outb_p(E8390_NODMA+E8390_PAGE0+E8390_START, e8390_base);
		outb_p(E8390_TXCONFIG, e8390_base + EN0_TXCR); /* xmit on. */
		/* 3c503 TechMan says rxconfig only after the NIC is started. */
		outb_p(E8390_RXCONFIG,	e8390_base + EN0_RXCR); /* rx on,  */
		dev->set_multicast_list(dev);		/* Get the multicast status right if this
							   was a reset. */
    }
    return;
}

/* Trigger a transmit start, assuming the length is valid. */
static void NS8390_trigger_send(struct device *dev, unsigned int length,
								int start_page)
{
    int e8390_base = dev->base_addr;
    
    outb_p(E8390_NODMA+E8390_PAGE0, e8390_base);
    
    if (inb_p(e8390_base) & E8390_TRANS) {
		printk("%s: trigger_send() called with the transmitter busy.\n",
			   dev->name);
		return;
    }
    outb_p(length & 0xff, e8390_base + EN0_TCNTLO);
    outb_p(length >> 8, e8390_base + EN0_TCNTHI);
    outb_p(start_page, e8390_base + EN0_TPSR);
    outb_p(E8390_NODMA+E8390_TRANS+E8390_START, e8390_base);
    return;
}

/*
 * Local variables:
 *  compile-command: "gcc -D__KERNEL__ -I/usr/src/linux/net/inet -Wall -Wstrict-prototypes -O6 -m486 -c 8390.c"
 *  version-control: t
 *  kept-new-versions: 5
 *  c-indent-level: 4
 *  tab-width: 4
 * End:
 */
