/* 
 <Beta>
=========================================================================
rtl8169.c: A RealTek Gigabit Ethernet driver for Linux kernel 2.2.x. 
 --------------------------------------------------------------------

 History:
 Feb 4 2002, ShuChen - created initially.
 	
 --------------------------------------------------------------------

 Copyright 2002 by Realtek Semiconductor Corp.
=========================================================================
 The link speed and duplex mode can be forced in 5 modes.
	 Command:	"insmod rtl8169 media = SET_MEDIA"
	 Ex:		"insmod rtl8169 media = 0x04" will force link mode to be 100Mpbs Half-duplex.
	
	The value of SET_MEDIA can be:
 		_10_Half	= 0x01
 		_10_Full	= 0x02
 		_100_Half	= 0x04
 		_100_Full	= 0x08
 		_1000_Full	= 0x10
  
=========================================================================
*/

#define RTL8169_VERSION "1.0"
#define MODNAME "RTL8169"
#define RTL8169_DRIVER_NAME   MODNAME " Gigabit Ethernet driver " RTL8169_VERSION


#ifndef USE_MEM_OPS
/* Note: Register access width and timing restrictions apply in MMIO mode.
   This updated driver should nominally work, but I/O mode is better tested. */
#define USE_IO_OPS
#endif


/* media options */
#define MAX_UNITS 8
static int media[MAX_UNITS] = {-1, -1, -1, -1, -1, -1, -1, -1};


/* The user-configurable values.
   These may be modified when a driver module is loaded.*/
static int debug = 1;			/* 1 normal messages, 0 quiet .. 7 verbose. */
/* Maximum events (Rx packets, etc.) to handle at each interrupt. */
static int max_interrupt_work = 20;

/* Maximum number of multicast addresses to filter (vs. Rx-all-multicast).
   The RTL chips use a 64 element hash table based on the Ethernet CRC.  It
   is efficient to update the hardware filter, but recalculating the table
   for a long filter list is painful.  */
static int multicast_filter_limit = 32;

/* Operational parameters that usually are not changed. */
/* Time in jiffies before concluding the transmitter is hung. */
#define TX_TIMEOUT  (6*HZ)

#ifndef __KERNEL__
#define __KERNEL__
#endif

#if !defined(__OPTIMIZE__)
#warning  You must compile this file with the correct options!
#warning  See the last lines of the source file.
#error You must compile this driver with "-O".
#endif

#include <linux/config.h>
#if defined(CONFIG_SMP) && ! defined(__SMP__)
#define __SMP__
#endif

#if defined(CONFIG_MODVERSIONS) && ! defined(MODVERSIONS)
#define MODVERSIONS
#endif

#include <linux/version.h>
#include <linux/module.h>
#if LINUX_VERSION_CODE < 0x20300  &&  defined(MODVERSIONS)
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <asm/processor.h>		/* Processor type for cache alignment. */
#include <asm/bitops.h>
#include <asm/io.h>

#if LINUX_VERSION_CODE >= 0x20300
#include <linux/spinlock.h>
#elif LINUX_VERSION_CODE >= 0x20200
#include <asm/spinlock.h>
#endif

#ifdef INLINE_PCISCAN
#include "k_compat.h"
#else
#include "kern_compat.h"
#endif

#if (LINUX_VERSION_CODE >= 0x20100)  &&  defined(MODULE)
char kernel_version[] = UTS_RELEASE;
#endif



static void *rtl8169_probe1(struct pci_dev *pdev, void *init_dev, long ioaddr, int irq, int chip_idx, int find_cnt);
//static int rtl_pwr_event(void *dev_instance, int event);

enum chip_capability_flags {HAS_MII_XCVR=0x01, HAS_CHIP_XCVR=0x02, HAS_LNK_CHNG=0x04};
#ifdef USE_IO_OPS
#define RTL8169_IOTYPE  PCI_USES_MASTER|PCI_USES_IO |PCI_ADDR0
#else
#define RTL8169_IOTYPE  PCI_USES_MASTER|PCI_USES_MEM|PCI_ADDR1
#endif

#define RTL8169_CAPS  HAS_CHIP_XCVR|HAS_LNK_CHNG

static struct pci_id_info pci_tbl[] = {
	{"RealTek RTL8169 Fast Ethernet", { 0x816910ec, 0xffffffff,},
	 RTL8169_IOTYPE, 0x80, RTL8169_CAPS, },
	{0,},						/* 0 terminated list. */
};

struct drv_id_info rtl8169_drv_id = {
	"realtek", PCI_HOTSWAP, PCI_CLASS_NETWORK_ETHERNET<<8, pci_tbl,
	rtl8169_probe1, /*rtl_pwr_event*/ NULL };

#ifndef USE_IO_OPS
#undef inb
#undef inw
#undef inl
#undef outb
#undef outw
#undef outl
#define inb readb
#define inw readw
#define inl readl
#define outb writeb
#define outw writew
#define outl writel
#endif


#define MAC_ADDR_LEN	6		/* MAC address length*/

#define NUM_TX_DESC	64			/* Number of Tx descriptor registers for rtl8169*/
#define NUM_RX_DESC	64			/* Number of Rx descriptor registers for rtl8169*/
#define RX_BUF_SIZE	1536		/* Rx Buffer size */

#define TX_FIFO_THRESH 256			/* In bytes, rounded down to 32 byte units. */
#define RX_FIFO_THRESH	7			/* 7 means NO threshold, Rx buffer level before first PCI xfer.  */
#define RX_DMA_BURST	6			/* Maximum PCI burst, '6' is 1024 */
#define TX_DMA_BURST	6			/* Maximum PCI burst, '6' is 1024 */
#define EarlyTxThld 	0x3F		/* 0x3F means NO early transmit */
#define RxPacketMaxSize	0x0800		/* Maximum size supported is 16K-1 */
#define InterFrameGap	0x03		/* 3 means InterFrameGap = the shortest one */


/* Symbolic offsets to registers. */
enum RTL8169_registers {
	MAC0 = 0,		/* Ethernet hardware address. */
	MAR0 = 8,		/* Multicast filter. */
	TxDescStartAddr	= 0x20,
	TxHDescStartAddr= 0x28,
	FLASH	= 0x30,
	ERSR	= 0x36,
	ChipCmd	= 0x37,
	TxPoll	= 0x38,
	IntrMask = 0x3C,
	IntrStatus = 0x3E,
	TxConfig = 0x40,
	RxConfig = 0x44,
	RxMissed = 0x4C,
	Cfg9346 = 0x50,
	Config0	= 0x51,
	Config1	= 0x52,
	Config2	= 0x53,	
	Config3	= 0x54,
	Config4	= 0x55,	
	Config5	= 0x56,
	MultiIntr = 0x5C,
	PHYAR	= 0x60,
	TBICSR	= 0x64,
	TBI_ANAR = 0x68,
	TBI_LPAR = 0x6A,
	PHYstatus = 0x6C,
	RxMaxSize = 0xDA,
	CPlusCmd = 0xE0,
	RxDescStartAddr	= 0xE4,
	EarlyTxThres	= 0xEC,
	FuncEvent	= 0xF0,
	FuncEventMask	= 0xF4,
	FuncPresetState	= 0xF8,
	FuncForceEvent	= 0xFC,		

	IntMitiR	= 0xE2,
};

enum InterruptStatusBits {
	SYSErr 		= 0x8000,
	PCSTimeout	= 0x4000,
	SWInt		= 0x0100,
	TxDescUnavail	= 0x80,
	RxFIFOOver 	= 0x40,
	RxUnderrun 	= 0x20,
	RxOverflow 	= 0x10,
	TxErr 	= 0x08,
	TxOK 	= 0x04,
	RxErr 	= 0x02,
	RxOK 	= 0x01,
};

enum RxStatusDesc {
	RxRES = 0x00200000,
	RxCRC = 0x00080000,
	RxRUNT= 0x00100000,
	RxRWT = 0x00400000,
};


enum ChipCmdBits {
	CmdReset = 0x10,
	CmdRxEnb = 0x08,
	CmdTxEnb = 0x04,
	RxBufEmpty = 0x01,
};


enum Cfg9346Bits {
	Cfg9346_Lock = 0x00,
	Cfg9346_Unlock = 0xC0,
};


enum rx_mode_bits {
	AcceptErr = 0x20,
	AcceptRunt = 0x10,
	AcceptBroadcast = 0x08,
	AcceptMulticast = 0x04,
	AcceptMyPhys = 0x02,
	AcceptAllPhys = 0x01,
};


/* Bits in RxConfig. */
enum RxConfigBits {
	RxCfgFIFOShift = 13,
	RxCfgDMAShift = 8,
};

/* Bits in TxConfig. */
enum TxConfigBits {
	TxInterFrameGapShift = 24,
	TxDMAShift = 8,		/* DMA burst value (0-7) is shift this many bits */
};


enum rtl8169_PHYstatus {
	TBI_Enable	= 0x80,
	TxFlowCtrl	= 0x40,
	RxFlowCtrl	= 0x20,
	_1000bpsF	= 0x10,
	_100bps		= 0x08,
	_10bps		= 0x04,
	LinkStatus	= 0x02,
	FullDup		= 0x01,
};


enum GIGABIT_PHY_registers {
	PHY_CTRL_REG = 0,
	PHY_STAT_REG = 1,
	PHY_AUTO_NEGO_REG = 4,
	PHY_1000_CTRL_REG = 9,
};

enum GIGABIT_PHY_REG_BIT {
	//PHY_CTRL_REG = 0;
	PHY_Restart_Auto_Nego	= 0x0200,
	PHY_Enable_Auto_Nego	= 0x1000,
	
	//PHY_STAT_REG = 1;
	PHY_Auto_Neco_Comp	= 0x0020,

	//PHY_AUTO_NEGO_REG = 4;
	PHY_Cap_10_Half		= 0x0020,
	PHY_Cap_10_Full		= 0x0040,
	PHY_Cap_100_Half	= 0x0080,
	PHY_Cap_100_Full	= 0x0100,
	
	//PHY_1000_CTRL_REG = 9;
	PHY_Cap_1000_Full	= 0x0200,

	PHY_Cap_Null		= 0x0,
};


enum _MediaType{
	_10_Half	= 0x01,
	_10_Full	= 0x02,
	_100_Half	= 0x04,
	_100_Full	= 0x08,
	_1000_Full	= 0x10,
};

enum _TBICSRBit {
	TBILinkOK 	= 0x02000000,
};


enum _DescStatusBit {
	OWNbit	= 0x80000000,
	EORbit = 0x40000000,
	FSbit	= 0x20000000,
	LSbit	= 0x10000000,
};


struct TxDesc {
	u32		status;
	u32		vlan_tag;
	u32		buf_addr;
	u32		buf_Haddr;
};

struct RxDesc {
	u32		status;
	u32		vlan_tag;
	u32		buf_addr;
	u32		buf_Haddr;
};

#define PRIV_ALIGN	15 	/* Desired alignment mask */
struct rtl8169_private {
	
	//=============
	struct net_device *next_module;
	void *priv_addr;					/* Unaligned address for kfree */
	void *mmio_addr;							/* memory map physical address*/
	struct pci_dev *pci_dev;					/* Index of PCI device  */
	struct net_device_stats stats;				/* statistics of net device */
	unsigned int rx_config;
	spinlock_t lock;							/* spin lock flag */
	int chipset;
	int chip_id;
	unsigned long cur_rx;						/* Index into the Rx descriptor buffer of next Rx pkt. */
	unsigned long cur_tx;						/* Index into the Tx descriptor buffer of next Rx pkt. */
	unsigned long dirty_tx;
	unsigned long tx_full;				/* The Tx queue is full. */
	unsigned char	*TxDescArrays;				/* Index of Tx Descriptor buffer */
	unsigned char	*RxDescArrays;				/* Index of Rx Descriptor buffer */
	struct	TxDesc	*TxDescArray;				/* Index of 256-alignment Tx Descriptor buffer */
	struct	RxDesc	*RxDescArray;				/* Index of 256-alignment Rx Descriptor buffer */
	unsigned char	*RxBufferRings;				/* Index of Rx Buffer  */
	unsigned char	*RxBufferRing[NUM_RX_DESC];	/* Index of Rx Buffer array */
	struct	sk_buff	*Tx_skbuff[NUM_TX_DESC];	/* Index of Transmit data buffer */

};

#ifdef MODULE
#if LINUX_VERSION_CODE > 0x20115
MODULE_AUTHOR("Realtek");
MODULE_DESCRIPTION("RealTek RTL8169 Gigabit Ethernet driver");
MODULE_PARM (media, "1-" __MODULE_STRING(MAX_UNITS) "i");
#endif
#endif


static int rtl8169_open(struct net_device *dev);
static int rtl8169_init_ring (struct net_device *dev);
static void rtl8169_hw_start (struct net_device *dev);
static int rtl8169_start_xmit(struct sk_buff *skb, struct net_device *dev);
static void rtl8169_tx_interrupt (struct net_device *dev, struct rtl8169_private *priv, long ioaddr);
static void rtl8169_rx_interrupt (struct net_device *dev, struct rtl8169_private *priv, long ioaddr);
static void rtl8169_interrupt(int irq, void *dev_instance, struct pt_regs *regs);
static void rtl8169_tx_clear (struct rtl8169_private *tp);
static int rtl8169_close(struct net_device *dev);
static struct net_device_stats *rtl8169_get_stats(struct net_device *dev);
static inline u32 ether_crc(int length, unsigned char *data);
static void set_rx_mode(struct net_device *dev);
static void rtl8169_soft_reset(struct net_device *dev);


static const unsigned int rtl8169_tx_config = (TX_DMA_BURST << TxDMAShift) | (InterFrameGap << TxInterFrameGapShift) ;
static const unsigned int rtl8169_rx_config = (RX_FIFO_THRESH << RxCfgFIFOShift) | (RX_DMA_BURST << RxCfgDMAShift) ;
static const short int rtl8169_intr_mask = SYSErr | PCSTimeout | RxUnderrun | RxOverflow | RxFIFOOver |
									TxErr | TxOK | RxErr | RxOK ;


/* A list of all installed RTL8129 devices, for removing the driver module. */
static struct net_device *root_rtl8169_dev = NULL;


//#define RTL8169_DEBUG

#ifdef RTL8169_DEBUG
#  define RTL8169_DEBUG_PRINT_FUNC() printk(KERN_DEBUG "%s\n" , __FUNCTION__ )
#else
#  define RTL8169_DEBUG_PRINT_FUNC()	{}
#endif






//======================================================================================================
void udelay(int cnt)
{
	int i=cnt,j;
	while(--i>0){
		for(j=1500;j>0;j--){}
	}
}





//======================================================================================================
static void rtl8169_soft_reset(struct net_device *dev)
{
	int	i;
	long	ioaddr = dev->base_addr;
		
	outb(CmdReset, ioaddr + ChipCmd);

	// Check that the chip has finished the reset.
	for (i = 10000; i > 0; i--){
		if ( inb(ioaddr + ChipCmd) == 0)break;
		else udelay(1);
	}
}






//=================================================================
//	PHYAR
//	bit		Symbol
//	31		Flag
//	30-21	reserved
//	20-16	5-bit GMII/MII register address
//	15-0	16-bit GMII/MII register data
//=================================================================
void WRITE_GMII_REG( long ioaddr, int RegAddr, int value )
{
	int	i;
	
	outl(0x80000000 | (RegAddr&0xFF)<<16 | value, ioaddr + PHYAR);
	udelay(1000);

	for( i = 2000; i > 0 ; i -- ){
		// Check if the RTL8169 has completed writing to the specified MII register
		if( ! ( inl(ioaddr+PHYAR) & 0x80000000 ) ){
			break;
		}
		else{
			udelay(100);
		}
	}// end of for() loop
//	return ( i!=0 ) ? 1 : 0 ;
}
//=================================================================
int READ_GMII_REG( long ioaddr, int RegAddr )
{ 
	int i, value = -1;
	
	outl(0x0 | (RegAddr&0xFF)<<16, ioaddr + PHYAR);
	udelay(1000);

	for( i = 2000; i > 0 ; i -- ){
		// Check if the RTL8169 has completed retrieving data from the specified MII register
		if( inl(ioaddr + PHYAR) & 0x80000000 ){
			value = (int)( inl(ioaddr + PHYAR)&0xFFFF );
			break;
		}
		else{
			udelay(100);
		}
	}// end of for() loop
	return value;
}







//======================================================================================================
static void *rtl8169_probe1(struct pci_dev *pdev, void *init_dev,
							long ioaddr, int irq, int chip_idx, int found_cnt)
{
	struct net_device *dev;
	struct rtl8169_private *np;
	void *priv_mem;
	int i;
	unsigned long tmp;
	int option = -1, Cap10_100 = 0, Cap1000 = 0;
	static int board_idx = -1;

	RTL8169_DEBUG_PRINT_FUNC();
	board_idx++;

	dev = init_etherdev(init_dev, 0);

	printk(KERN_INFO "%s: %s at %#lx, IRQ=%d, HWaddr=", dev->name, pci_tbl[chip_idx].name, ioaddr, irq);

	for(i=0;i<6;i++){
		dev->dev_addr[i] = inb(ioaddr+MAC0+i);
	}

	for(i=0;i<5;i++){
		printk("%2.2x:", dev->dev_addr[i] );
	}
	printk("%2.2x\n", dev->dev_addr[i] );


	// Make certain elements e.g. descriptor lists are aligned. //
	priv_mem = kmalloc(sizeof(*np) + PRIV_ALIGN, GFP_KERNEL);
	/* Check for the very unlikely case of no memory. */
	if (priv_mem == NULL)
		return NULL;

	/* We do a request_region() to register /proc/ioports info. */
	request_region(ioaddr, pci_tbl[chip_idx].io_size, dev->name);

	dev->base_addr = ioaddr;
	dev->irq = irq;

	dev->priv = np = (void *)(((long)priv_mem + PRIV_ALIGN) & ~PRIV_ALIGN);
	memset(np, 0, sizeof(*np));
	np->priv_addr = priv_mem;

	np->next_module = root_rtl8169_dev;
	root_rtl8169_dev = dev;

	np->pci_dev = pdev;
	np->chip_id = chip_idx;

	// The Rtl8169-specific entries in the device structure. 
	dev->open = &rtl8169_open;
	dev->hard_start_xmit = &rtl8169_start_xmit;
	dev->stop = &rtl8169_close;
	dev->get_stats = &rtl8169_get_stats;
	dev->set_multicast_list = &set_rx_mode;
	//========================================================================
	// Soft reset the chip.
	rtl8169_soft_reset(dev);

	// Identify chip attached to board
	tmp = inl(ioaddr + TxConfig);
	tmp = ( (tmp&0x7c000000) + ( (tmp&0x00800000)<<2 ) )>>24;
	if (tmp == 0) {
		np->chipset = 1;
	}
	printk (KERN_INFO "%s: Identified chip type is '%s'.\n", 
						dev->name, (np->chipset != 0) ? "RTL-8169" : "Unknown");

	//========================================================================
	//	Report Phy Status (GMII or TBI)
	//========================================================================
/*
	{ 
		u8hyStatus;
		Phtatus = inb(ioaddr + PHYstatus);
		ifPhyStatus & _1000bpsF ){
			printk(KERN_INFO"%s: Link Status = %s, PHY Status = 1000 Full-Duplex!\n",
					dev->name,
					(PhyStatus & LinkStatus) ? "OK" : "NoLink" );
		} 
		el{
			printk(KERN_INFO"%s: Link Status = %s, PHY Status = %sMbps %s-Duplex!\n",
					dev->name,
					(PhyStatus & LinkStatus) ? "OK" : "NoLink",
					(PhyStatus & _10bps) ? "10" : "100",
					(PhyStatus & FullDup) ? "Full" : "Half"	);
		} 
	} 
*/
	//========================================================================
	//Set Link Type !
	//if(!TBI_Link)
	//	if(Force_Mode_Enabled){set auto-negotiation ability to the forced speed}
	//	else {set all of the auto-negotiation ability}
	//else
	//	Just check TBI link status
	//========================================================================
	if( !(inb(ioaddr + PHYstatus) & TBI_Enable) ){
		option = media[board_idx];
		// Force RTL8169 in 10/100/1000 Full/Half mode.
		if( option > 0 ){
			printk(KERN_INFO "%s: Force-mode Enabled.\n", dev->name);
			switch( option ){
				case _10_Half: 	Cap10_100 = PHY_Cap_10_Half;	Cap1000 = PHY_Cap_Null;		break;
				case _10_Full: 	Cap10_100 = PHY_Cap_10_Full;	Cap1000 = PHY_Cap_Null;		break;
				case _100_Half: Cap10_100 = PHY_Cap_100_Half;	Cap1000 = PHY_Cap_Null;		break;
				case _100_Full: Cap10_100 = PHY_Cap_100_Full;	Cap1000 = PHY_Cap_Null; 	break;
				case _1000_Full:Cap10_100 = PHY_Cap_Null;	Cap1000 = PHY_Cap_1000_Full;	break;
				default: break;
			}
			WRITE_GMII_REG( ioaddr, PHY_AUTO_NEGO_REG, Cap10_100 );
			WRITE_GMII_REG( ioaddr, PHY_1000_CTRL_REG, Cap1000 );
		}
		else{
			printk(KERN_INFO "%s: Auto-negotiation Enabled.\n", dev->name);
	
			// enable 10/100 Full/Half Mode
			WRITE_GMII_REG( ioaddr, PHY_AUTO_NEGO_REG, PHY_Cap_10_Half | PHY_Cap_10_Full | PHY_Cap_100_Half | PHY_Cap_100_Full );
	
			// enable 1000 Full Mode
			WRITE_GMII_REG( ioaddr, PHY_1000_CTRL_REG, PHY_Cap_1000_Full );
	
		}// end of if( option > 0 )

		// Enable auto-negotiation and restart auto-nigotiation
		WRITE_GMII_REG( ioaddr, PHY_CTRL_REG, PHY_Enable_Auto_Nego | PHY_Restart_Auto_Nego );

		// wait for auto-negotiation process
		for( i = 2000; i > 0; i-- ){
			//check if auto-negotiation complete
			if( READ_GMII_REG(ioaddr, PHY_STAT_REG) & PHY_Auto_Neco_Comp ){
				udelay(100);
				option = inb(ioaddr + PHYstatus);
				if( option & _1000bpsF ){
					printk(KERN_INFO "%s: 1000Mbps Full-duplex Link %s!\n",
						dev->name,
						( inb(ioaddr+PHYstatus)&LinkStatus )?"OK":"Failed");
				}
				else{
					printk(KERN_INFO "%s: %sMbps %s-duplex Link %s!\n",
						dev->name,
						(option & _100bps) ? "100" : "10",
						(option & FullDup) ? "Full" : "Half",
						( inb(ioaddr+PHYstatus)&LinkStatus )?"OK":"Failed" );
				}
				break;
			}
			else{
				udelay(100);
			}// end of if( READ_GMII_REG(ioaddr, 1) & 0x20 )
		}// end for-loop to wait for auto-negotiation process

		if(i==0)printk(KERN_INFO"%s: Auto-negotiation process is not completed!\n", dev->name);
	}// end of TBI is not enabled
	else{	
		udelay(100);
		printk(KERN_INFO "%s: 1000Mbps Full-duplex operation, TBI Link %s!\n",
					dev->name, (inl(ioaddr + TBICSR) & TBILinkOK) ? "OK" : "Failed" );
	}// end of TBI is not enabled
	//========================================================================

	return dev;
}














//======================================================================================================
static int rtl8169_init_ring (struct net_device *dev)
{
	struct rtl8169_private *priv = (struct rtl8169_private *)dev->priv;
	int	diff;
	int i;
	u32 TxPhyAddr, RxPhyAddr;


	RTL8169_DEBUG_PRINT_FUNC();


	priv->cur_rx = 0;
	priv->cur_tx = 0;
	priv->dirty_tx = 0;
	priv->tx_full = 0;

	//===============================================================================
	// Allocate memory for Tx Desscriptors.
	// Make sure memory address to be 256 bytes alignment.
	//===============================================================================
	priv->TxDescArrays = kmalloc( NUM_TX_DESC * sizeof(struct TxDesc)+256 , GFP_KERNEL );
	if ( priv->TxDescArrays == NULL ) {
		printk(KERN_INFO"Allocate TxDescArray failed\n");
		free_irq(dev->irq, dev);
		return -ENOMEM;
	}
	TxPhyAddr = virt_to_bus(priv->TxDescArrays);
	diff = 256 - (TxPhyAddr-((TxPhyAddr >> 8)<< 8));
	priv->TxDescArray = (struct TxDesc *)(priv->TxDescArrays + diff);

	//===============================================================================
	// Allocate memory for Rx Desscriptors.
	// Make sure memory address to be 256 bytes alignment.
	//===============================================================================
	priv->RxDescArrays = kmalloc( NUM_RX_DESC * sizeof(struct RxDesc)+256 , GFP_KERNEL );
	if ( priv->TxDescArrays == NULL || priv->RxDescArrays == NULL ) {
		printk(KERN_INFO"Allocate RxDescArray failed\n");
		free_irq(dev->irq, dev);
		if( priv->TxDescArrays ) kfree(priv->TxDescArrays);
		return -ENOMEM;
	}
	RxPhyAddr = virt_to_bus(priv->RxDescArrays);
	diff = 256 - (RxPhyAddr-((RxPhyAddr >> 8)<< 8));
	priv->RxDescArray = (struct RxDesc *)(priv->RxDescArrays + diff);
	

	//===============================================================================
	// Allocate memory for Rx buffer.
	//===============================================================================
	priv->RxBufferRings = kmalloc( RX_BUF_SIZE * NUM_RX_DESC, GFP_KERNEL );
	if( priv->RxBufferRings == NULL ){
		printk( KERN_INFO"Allocate RxBufferRing failed\n" );
		free_irq(dev->irq, dev);
		if( priv->TxDescArrays ) kfree(priv->TxDescArrays);
		if( priv->RxDescArrays ) kfree(priv->RxDescArrays);
		return -ENOMEM;
	}


	//===============================================================================
	// Reset Tx and Rx Desscriptors.
	//===============================================================================
	memset( priv->TxDescArray, 0x0, NUM_TX_DESC *sizeof(struct TxDesc) );
	memset( priv->RxDescArray, 0x0, NUM_RX_DESC *sizeof(struct RxDesc) );

	//===============================================================================
	// Reset Tx skb buffer.
	//===============================================================================
	for (i=0 ; i<NUM_TX_DESC ; i++){
		priv->Tx_skbuff[i] = NULL;
	}

	//===============================================================================
	// Fill Rx descripter in advance
	//===============================================================================
	for (i=0; i <NUM_RX_DESC; i++) {
		if( i == (NUM_RX_DESC-1) )
			priv->RxDescArray[i].status = (OWNbit | EORbit) + RX_BUF_SIZE;
		else
			priv->RxDescArray[i].status = OWNbit + RX_BUF_SIZE;

		priv->RxBufferRing[i] = &(priv->RxBufferRings[i*RX_BUF_SIZE]);
		priv->RxDescArray[i].buf_addr = virt_to_bus(priv->RxBufferRing[i]);
	}

	return 0;

}









//======================================================================================================
/* Start the hardware at open or resume. */
static void rtl8169_hw_start (struct net_device *dev)
{

	struct rtl8169_private *priv = (struct rtl8169_private *)dev->priv;
	long ioaddr = dev->base_addr;


	RTL8169_DEBUG_PRINT_FUNC();


	// Soft reset the chip.
	rtl8169_soft_reset(dev);


	priv->cur_rx = 0;

	outb(Cfg9346_Unlock, ioaddr + Cfg9346);
	outb(CmdRxEnb | CmdTxEnb, ioaddr + ChipCmd);
	outb(EarlyTxThld, ioaddr + EarlyTxThres);

//++++++++++++
	outw(0x1118, ioaddr + IntMitiR);

	// For gigabit rtl8169
	outw(RxPacketMaxSize, ioaddr + RxMaxSize);

	// Set Rx Config register
	outl(rtl8169_rx_config | ( inl( ioaddr + RxConfig ) & 0xff7e1880 ), ioaddr + RxConfig);
	
	priv->rx_config = rtl8169_rx_config;
	
	// Set DMA burst size and Interframe Gap Time //
	outl(rtl8169_tx_config, ioaddr + TxConfig);

	outl(virt_to_bus(priv->TxDescArray), ioaddr + TxDescStartAddr);
	outl(virt_to_bus(priv->RxDescArray), ioaddr + RxDescStartAddr);
	
	outb(Cfg9346_Lock, ioaddr + Cfg9346);

	outl(0, ioaddr + RxMissed);

	set_rx_mode (dev);

	outw(inw(ioaddr + MultiIntr) & 0xF000, ioaddr + MultiIntr);

	// Enable all known interrupts by setting the interrupt mask. //
	outw(rtl8169_intr_mask, ioaddr + IntrMask);

//	netif_start_queue (dev);

}//end of rtl8169_hw_start (struct net_device *dev)









//======================================================================================================
static int rtl8169_open(struct net_device *dev)
{

	RTL8169_DEBUG_PRINT_FUNC();


	// Soft reset the chip. 
	rtl8169_soft_reset(dev);


	MOD_INC_USE_COUNT;
	if (request_irq(dev->irq, &rtl8169_interrupt, SA_SHIRQ, dev->name, dev)) {
		MOD_DEC_USE_COUNT;
		return -EAGAIN;
	}

	rtl8169_init_ring (dev);
	rtl8169_hw_start (dev);

	dev->tbusy = 0;


	return 0;
}








//======================================================================================================
static int rtl8169_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct rtl8169_private *priv = (struct rtl8169_private *)dev->priv;
	long ioaddr = dev->base_addr;
	int entry = priv->cur_tx % NUM_TX_DESC;


	RTL8169_DEBUG_PRINT_FUNC();

	if (test_and_set_bit(0, (void*)&dev->tbusy) != 0) {
		if (jiffies - dev->trans_start >= TX_TIMEOUT){
			//-------------------------------//
			// reset chip and restart driver //
			//-------------------------------//
			//rtl8129_tx_timeout(dev);
		}
		return 1;
	}


	if( (priv->TxDescArray[entry].status & OWNbit)==0 ){
		priv->Tx_skbuff[entry] = skb;	
		priv->TxDescArray[entry].buf_addr = virt_to_bus(skb->data);
		if( entry != (NUM_TX_DESC-1) ){
			priv->TxDescArray[entry].status = (OWNbit | FSbit | LSbit) | ( (skb->len > ETH_ZLEN) ? skb->len : ETH_ZLEN);
		}
        	else{
		 	priv->TxDescArray[entry].status = (OWNbit | EORbit | FSbit | LSbit) | ( (skb->len > ETH_ZLEN) ? skb->len : ETH_ZLEN);
		}
	
		outb(0x40, ioaddr + TxPoll);	//set polling bit

		if (++priv->cur_tx - priv->dirty_tx < NUM_TX_DESC){
			clear_bit(0, (void*)&dev->tbusy);  /* Typical path */
		}
		else {
			set_bit(0, &priv->tx_full);
			if (priv->cur_tx - (volatile int)priv->dirty_tx < NUM_TX_DESC) {
				clear_bit(0, (void*)&dev->tbusy);
				clear_bit(0, &priv->tx_full);
			}
		}

		dev->trans_start = jiffies;

	}//end of if( (priv->TxDescArray[entry].status & 0x80000000)==0 )

	return 0;
}








//======================================================================================================
static void rtl8169_tx_interrupt (struct net_device *dev, struct rtl8169_private *priv, long ioaddr)
{

	unsigned long tx_left=0, dirty_tx = priv->dirty_tx;
	int entry = priv->cur_tx % NUM_TX_DESC;


	RTL8169_DEBUG_PRINT_FUNC();

	tx_left = priv->cur_tx - dirty_tx;

	while (tx_left > 0) {					
		if( (priv->TxDescArray[entry].status & OWNbit) == 0 ){
			dev_free_skb(priv->Tx_skbuff[dirty_tx % NUM_TX_DESC]);
			priv->Tx_skbuff[dirty_tx % NUM_TX_DESC] = NULL;
			if (test_bit(0, &priv->tx_full)) {
				/* The ring is no longer full, clear tbusy. */
				clear_bit(0, &priv->tx_full);
				clear_bit(0, (void*)&dev->tbusy);
				netif_wake_queue(dev);
			}
			priv->stats.tx_packets++;			
			dirty_tx++;				
			tx_left--;
			entry++;
		}
	}						

	if (priv->dirty_tx != dirty_tx) {			
		priv->dirty_tx = dirty_tx;			
		netif_wake_queue (dev);			
	}							

}







//======================================================================================================
static void rtl8169_rx_interrupt (struct net_device *dev, struct rtl8169_private *priv, long ioaddr)
{

	struct sk_buff *skb; 					
	int pkt_size = 0 ;					
	int cur_rx = priv->cur_rx;

	RTL8169_DEBUG_PRINT_FUNC();


	while ( (priv->RxDescArray[cur_rx].status & OWNbit)== 0) {
		
	    if( priv->RxDescArray[cur_rx].status & RxRES ){
		printk(KERN_INFO "%s: Rx ERROR!!!\n", dev->name);
		priv->stats.rx_errors++;
		if (priv->RxDescArray[cur_rx].status & (RxRWT|RxRUNT) ){
			priv->stats.rx_length_errors++;
		}
		if (priv->RxDescArray[cur_rx].status & RxCRC){
			priv->stats.rx_crc_errors++;
		}
	    }
	    else{
   		pkt_size=(int)(priv->RxDescArray[cur_rx].status & 0x00001FFF)-4;
		skb = dev_alloc_skb(pkt_size + 2);
		if (skb != NULL) {
		  	skb->dev = dev;
			skb_reserve (skb, 2);	// 16 byte align the IP fields. //
			eth_copy_and_sum(skb, priv->RxBufferRing[cur_rx], pkt_size, 0);
			skb_put(skb, pkt_size);
			skb->protocol = eth_type_trans(skb, dev);
			netif_rx (skb);

			if( cur_rx == (NUM_RX_DESC-1) ){
				priv->RxDescArray[cur_rx].status  = (OWNbit | EORbit) + RX_BUF_SIZE;
			}
			else{
				priv->RxDescArray[cur_rx].status  = OWNbit + RX_BUF_SIZE;
			}
			priv->RxDescArray[cur_rx].buf_addr = virt_to_bus( priv->RxBufferRing[cur_rx] );
			dev->last_rx = jiffies;
    			priv->stats.rx_bytes += pkt_size;
    			priv->stats.rx_packets++;				
		}
		else{
			printk(KERN_WARNING"%s: Memory squeeze, deferring packet.\n",dev->name);
			priv->stats.rx_dropped++;
		}// end of if (skb != NULL)
	    }// end of if( priv->RxDescArray[cur_rx].status & RxRES )

	    cur_rx = (cur_rx +1) % NUM_RX_DESC;

	}// end of while ( (priv->RxDescArray[cur_rx].status & 0x80000000)== 0)

	priv->cur_rx = cur_rx;					
}






//======================================================================================================
/* The interrupt handler does all of the Rx thread work and cleans up after the Tx thread. */
static void rtl8169_interrupt(int irq, void *dev_instance, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_instance;
	struct rtl8169_private *priv = (struct rtl8169_private *)dev->priv;
	int boguscnt = max_interrupt_work;
	long ioaddr = dev->base_addr;
	int status = 0;


	RTL8169_DEBUG_PRINT_FUNC();

/*
	if (dev->interrupt) {
		printk(KERN_ERR "%s: Re-entering the interrupt handler.\n", dev->name);
		return;
	}
	dev->interrupt = 1;
*/

	do {
		status = inw(ioaddr + IntrStatus);

		// h/w no longer present (hotplug?) or major error, bail //
		if (status == 0xFFFF)
			break;

//		if (status & RxUnderrun)
//			link_changed = RTL_R16 (CSCR) & CSCR_LinkChangeBit;

		outw((status & RxFIFOOver) ? (status | RxOverflow) : status, ioaddr + IntrStatus);

		if ( (status & ( SYSErr | PCSTimeout | RxUnderrun | RxOverflow | RxFIFOOver | TxErr | TxOK | RxErr | RxOK) ) == 0 )
			break;

		
		// Rx interrupt 
		if (status & (RxOK | RxUnderrun | RxOverflow | RxFIFOOver)){
			rtl8169_rx_interrupt (dev, priv, ioaddr);
		}

		// Tx interrupt
		if (status & (TxOK | TxErr)) {
			rtl8169_tx_interrupt (dev, priv, ioaddr);
		}

		boguscnt--;
	} while (boguscnt > 0);

	if (boguscnt <= 0) {
		printk (KERN_WARNING "%s: Too much work at interrupt!\n", dev->name);
		// Clear all interrupt sources.//
		outw(0xFFFF, ioaddr + IntrStatus);
	}

	return;
}







//======================================================================================================
static void rtl8169_tx_clear (struct rtl8169_private *tp)
{
	int i;


	RTL8169_DEBUG_PRINT_FUNC();


	tp->cur_tx = 0;
	for ( i = 0 ; i < NUM_TX_DESC ; i++ ){
		if ( tp->Tx_skbuff[i] != NULL ) {
			dev_kfree_skb ( tp->Tx_skbuff[i] );
			tp->Tx_skbuff[i] = NULL;
			tp->stats.tx_dropped++;
		}
	}

}






//======================================================================================================
static int rtl8169_close(struct net_device *dev)
{

	struct rtl8169_private *priv = (struct rtl8169_private *)dev->priv;
	long ioaddr = dev->base_addr;
	int i;


	RTL8169_DEBUG_PRINT_FUNC();


	dev->tbusy = 1;

	// Stop the chip's Tx and Rx DMA processes. //
	outb(0x00, ioaddr + ChipCmd);

	// Disable interrupts by clearing the interrupt mask. //
	outw(0x0000, ioaddr + IntrMask);

	// Update the error counts. //
	priv->stats.rx_missed_errors += inl(ioaddr + RxMissed);
	outl(0x00000000, ioaddr + RxMissed);

	free_irq (dev->irq, dev);

	rtl8169_tx_clear (priv);
	kfree(priv->TxDescArrays);
	kfree(priv->RxDescArrays);
	priv->TxDescArrays = NULL;
	priv->RxDescArrays = NULL;
	priv->TxDescArray = NULL;
	priv->RxDescArray = NULL;
	kfree(priv->RxBufferRings);
	for (i=0; i <NUM_RX_DESC; i++){
		priv->RxBufferRing[i] = NULL;
	}
	
	MOD_DEC_USE_COUNT;
	return 0;
}






//======================================================================================================
static struct net_device_stats *rtl8169_get_stats(struct net_device *dev)
{
	struct rtl8169_private *tp = (struct rtl8169_private *)dev->priv;
	long ioaddr = dev->base_addr;

	if (dev->start) {
		tp->stats.rx_missed_errors += inl(ioaddr + RxMissed);
		outl(0, ioaddr + RxMissed);
	}

	return &tp->stats;
}





//======================================================================================================
/* Set or clear the multicast filter for this adaptor.
   This routine is not state sensitive and need not be SMP locked. */

static unsigned const ethernet_polynomial = 0x04c11db7U;
static inline u32 ether_crc(int length, unsigned char *data)
{
    int crc = -1;

    while (--length >= 0) {
		unsigned char current_octet = *data++;
		int bit;
		for (bit = 0; bit < 8; bit++, current_octet >>= 1)
			crc = (crc << 1) ^
				((crc < 0) ^ (current_octet & 1) ? ethernet_polynomial : 0);
    }
    return crc;
}







//======================================================================================================
static void set_rx_mode(struct net_device *dev)
{
	struct rtl8169_private *tp = (struct rtl8169_private *)dev->priv;
	long ioaddr = dev->base_addr;
	u32 mc_filter[2];		 /* Multicast hash filter */
	int i, rx_mode;


	RTL8169_DEBUG_PRINT_FUNC();


	/* Note: do not reorder, GCC is clever about common statements. */
	if (dev->flags & IFF_PROMISC) {
		/* Unconditionally log net taps. */
		printk(KERN_NOTICE"%s: Promiscuous mode enabled.\n", dev->name);
		rx_mode = AcceptBroadcast|AcceptMulticast|AcceptMyPhys|AcceptAllPhys;
		mc_filter[1] = mc_filter[0] = 0xffffffff;
	} else if ((dev->mc_count > multicast_filter_limit)
			   || (dev->flags & IFF_ALLMULTI)) {
		/* Too many to filter perfectly -- accept all multicasts. */
		rx_mode = AcceptBroadcast | AcceptMulticast | AcceptMyPhys;
		mc_filter[1] = mc_filter[0] = 0xffffffff;
	} else {
		struct dev_mc_list *mclist;
		rx_mode = AcceptBroadcast | AcceptMulticast | AcceptMyPhys;
		mc_filter[1] = mc_filter[0] = 0;
		for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
			 i++, mclist = mclist->next)
			set_bit(ether_crc(ETH_ALEN, mclist->dmi_addr) >> 26, mc_filter);
	}
	/* We can safely update without stopping the chip. */
	outl(tp->rx_config | rx_mode, ioaddr + RxConfig);
	outl(mc_filter[0], ioaddr + MAR0 + 0);
	outl(mc_filter[1], ioaddr + MAR0 + 4);
	return;
}


/*
//======================================================================================================
static int rtl_pwr_event(void *dev_instance, int event)
{
	return 0;
}
*/



#ifdef MODULE
//======================================================================================================
int init_module(void)
{
	printk (KERN_INFO RTL8169_DRIVER_NAME " loaded\n");
	return pci_drv_register(&rtl8169_drv_id, NULL);
}


//======================================================================================================
void cleanup_module(void)
{
	struct net_device *next_dev;

	pci_drv_unregister(&rtl8169_drv_id);

	while (root_rtl8169_dev) {
		struct rtl8169_private *np = (void *)(root_rtl8169_dev->priv);
		unregister_netdev(root_rtl8169_dev);
		release_region(root_rtl8169_dev->base_addr, pci_tbl[np->chip_id].io_size);
#ifndef USE_IO_OPS
		iounmap((char *)(root_rtl8169_dev->base_addr));
#endif
		next_dev = np->next_module;
		if (np->priv_addr)
			kfree(np->priv_addr);
		kfree(root_rtl8169_dev);
		root_rtl8169_dev = next_dev;
	}
}

#endif  /* MODULE */


/*
 * Local variables:
 *  compile-command: "gcc -DMODULE -Wall -Wstrict-prototypes -O6 -c rtl8169.c"
 */
