
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBDSP     : Stellt elementare Routinen zur direkten Programmierung   }
{                  (also nicht ber Treiber wie CT-VOICE.DRV) des DSP (Di-  }
{                  gital Sound Processor) zur Verfgung. Einige der Routi-  }
{                  nen, wie zum Beispiel ReadDSP, wurden bereits in der     }
{                  Unit SBDrv implementiert.                                }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBDSP.PAS                                                }
{ entwickelt am  : 04.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.06                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBDSP;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

USES SBDrv;                                          { SVDrv-Unit einbinden }

CONST

  {-------------------------------------------------------------------------}
  { DSP-Kommandos                                                           }
  {-------------------------------------------------------------------------}

  { direkte Ausgabe ungepackter Daten }

  dsp_WriteDirectUnpack = $10;

  { DMA-Ausgabe ungepackter Daten mit Referenzbyte }

  dsp_WriteDMAUnpackRef = $14;

  { DMA-Ausgabe 2-Bits-gepackter Daten ohne Referenzbyte }

  dsp_WriteDMA2BitPackNoRef = $16;

  { DMA-Ausgabe 2-Bits-gepackter Daten mit Referenzbyte }

  dsp_WriteDMA2BitPackRef = $17;

  { DMA-Ausgabe 4-Bits-gepackter Daten ohne Referenzbyte }

  dsp_WriteDMA4BitPackNoRef = $74;

  { DMA-Ausgabe 4-Bits-gepackter Daten mit Referenzbyte }

  dsp_WriteDMA4BitPackRef = $75;

  { DMA-Ausgabe 2,6-Bits-gepackter Daten ohne Referenzbyte }

  dsp_WriteDMA26BitPackNoRef = $76;

  { DMA-Ausgabe 2,6-Bits-gepackter Daten mit Referenzbyte }

  dsp_WriteDMA26BitPackRef = $77;

  { direkte Eingabe ungepackter Daten }

  dsp_ReadDirectUnpack = $20;

  { DMA-Eingabe ungepackter Daten }

  dsp_ReadDMAUnpack = $24;

  { Setzen der sog. Zeitkonstante (TC = TIME CONSTANT), }
  { wodurch die Sample-Frequenz bestimmt wird           }

  dsp_SetTC = $40;

  { Programmieren einer Ruheperiode }

  dsp_SetSilence = $80;

  { Unterbrechung der DMA-bertragung }

  dsp_HaltDMATransfer = $D0;

  { Einschalten des Lautsprechers }

  dsp_SpeakerOn = $D1;

  { Ausschalten des Lautsprechers }

  dsp_SpeakerOff = $D3;

  { Lesen des Lautsprecher-Status    }
  { 00h = Lautsprecher ausgeschaltet }
  { FFh = Lautsprecher eingeschaltet }

  dsp_SpeakerStatus = $D8;

  { Fortsetzung der DMA-bertragung }

  dsp_ContinueDMATransfer = $D4;

  { Programmierung der Sound Blaster Pro Erweiterungen }
  { (sehr schnelle Soundausgabe und -eingabe)          }

  dsp_SBProExtensions = $48;

  { sehr schnelle 8-Bits-Soundausgabe (nur SB Pro) }

  dps_HighSpeedDMAWrite = $91;

  { sehr schnelle 8-Bits-Soundeingabe (nur SB Pro) }

  dsp_HighSpeedDMARead = $99;

{===========================================================================}
{ Prozedur InitDSP: Initialisiert den DSP durch den Aufruf der Routine      }
{                   CheckBaseAddress aus SBDrv.                             }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE InitDSP;

{===========================================================================}
{ Prozedur SetTimerFreq: Stellt eine Frequenz fr den Timer ein. Der Werte- }
{                        bereich liegt zwischen 19 und 50000 Hz.            }
{===========================================================================}
{ Eingabe: Freq = Timer-Frequenz in Hz (19..50000)                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetTimerFreq(Freq : Word);

{===========================================================================}
{ Prozedur SetOrigFreq: Stellt die Original-Frequenz des Timer von 18,2 Hz  }
{                       ein.                                                }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetOrigFreq;

{===========================================================================}
{ Prozedur WriteDSP: Sendet einen Bytewert an den Soundprozessor. Ist die-  }
{                    ser mit seiner letzten Aktion beschftigt, mu beim    }
{                    Senden des Bytewertes gewartet werden, bis diese Ak-   }
{                    tion ordungsgem beendet ist.                         }
{===========================================================================}
{ Eingabe: Value = Bytewert                                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE WriteDSP(Value : Byte);

{===========================================================================}
{ Prozedur WriteDirect: Veranlat den DSP zur Ausgabe von ungepackten       }
{                       Sounddaten im Direktmodus.                          }
{                                                                           }
{                       HINWEIS: Um eine gute VOICE-Ausgabe mit Hilfe die-  }
{                                ser Prozedur zu erreichen, kann man den    }
{                                Timer-Interrupt verwenden. Dabei mu man   }
{                                aber beachten, da dieser umprogrammiert   }
{                                werden mu, um eine hhere Ausgabe-Freq.   }
{                                als 18,2 mal in der Sekunde zu erreichen.  }
{===========================================================================}
{ Eingabe: Value = Bytewert                                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE WriteDirect(Value : Byte);

{===========================================================================}
{ Prozedur Speaker_On: Schaltet den Lautsprecher an.                        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Speaker_On;

{===========================================================================}
{ Prozedur Speaker_Off: Schaltet den Lautsprecher aus.                      }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Speaker_Off;

{===========================================================================}
{ Prozedur PlayDirect: Spielt einen Block von ungepackten Daten im Direkt-  }
{                      modus mit der angegebenen Ausgabe-Frequenz ab. Die   }
{                      Sounddaten mssen sich im Speicher befinden.         }
{===========================================================================}
{ Eingabe: DataPtr = Zeiger auf den Blockanfang                             }
{          Size    = Gre des Datenblocks                                  }
{          Freq    = Ausgabe-Frequenz in Hz (19..50000)                     }
{          Init    = TRUE, wenn der DSP vor der Soundausgabe initialisiert  }
{                    werden soll, sonst FALSE                               }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE PlayDirect(DataPtr : Pointer; Size : Word; Freq : Word;
  Init : Boolean);

{===========================================================================}
{ Prozedur Mixer: Stellt einen der Modi ein: Stereo oder Mono. Auerdem     }
{                 schaltet die Routine den Filter ein bzw. aus.             }
{                                                                           }
{                 ANMERKUNG: Die Funktion ist nur bei Sound Blaster Pro     }
{                            verfgbar.                                     }
{===========================================================================}
{ Eingabe: Stereo = TRUE, wenn Stereo gesetzt werden soll, FALSE fr Mono   }
{          Filter = TRUE, wenn ein Filter verwendet werden soll, sonst FAL- }
{                   SE                                                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Mixer(Stereo, Filter : Boolean);

IMPLEMENTATION

USES CRT, DOS;                                { CRT- und DOS-Unit einbinden }

VAR

  {-------------------------------------------------------------------------}
  { interne Variablen im Zusammenhang mit der Routine PlayDirect            }
  {-------------------------------------------------------------------------}

  BytePtr   : ^Byte;                                  { Zeiger auf ein Byte }
  Count_    : LongInt;                                         { ein Zhler }
  Done      : Boolean;         { TRUE, wenn die Soundausgabe zu beenden ist }
  BlockSize : Word;                  { Gre des abzuspielenden Datenblocks }

{===========================================================================}
{ Prozedur NewTimerInt: Neue Timer-Interruptroutine. ber sie wird eine     }
{                       Soundausgabe im Direktmodus realisiert.             }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE NewTimerInt; INTERRUPT;

BEGIN
  WriteDirect(BytePtr^);                          { Bytewert zum DSP senden }
  IF Count_ = BlockSize THEN                       { Datenblock abgespielt? }
    Done := TRUE                                 { ja, Soundausgabe beenden }
  ELSE                                                               { nein }
    BEGIN
      Inc(Count_);                                         { Zhler erhhen }
      BytePtr := Ptr(Seg(BytePtr^), Ofs(BytePtr^)+1); { nchstes Byte holen }
    END;
  Port[$20] := $20;                              { Interruptroutine beenden }
END;

{---------------------------------------------------------------------------}
{ Implementation von modulexternen Routinen                                 }
{---------------------------------------------------------------------------}

PROCEDURE InitDSP;

BEGIN
  CheckBaseAddress;
END;

PROCEDURE SetTimerFreq(Freq : Word);

VAR
  Timer : Word;                          { interner Wert fr Timer-Frequenz }

BEGIN

  { Prfen, ob die angegebene Frequenz im Wertebereich zwischen }
  { 19 und 50000 Hz liegt. Falls nicht, korrigieren.            }

  IF Freq < 19 THEN
    Freq := 19;
  IF Freq > 50000 THEN
    Freq := 50000;

  Port[$43] := $36;                                          { Steuerbefehl }
  Timer := 1193180 DIV Freq;   { internen Wert fr Timer-Frequenz berechnen }
  Port[$40] := Lo(Timer);              { Low-Byte fr den Timer-Wert setzen }
  Port[$40] := Hi(Timer);             { High-Byte fr den Timer-Wert setzen }
END;

PROCEDURE SetOrigFreq;

VAR
  Timer : Word;                          { interner Wert fr Timer-Frequenz }

BEGIN
  Port[$43] := $36;                                          { Steuerbefehl }
  Timer := $FFFF;               { Timer-Frequenz auf Originalwert (18,2 Hz) }
  Port[$40] := Lo(Timer);                 { Low-Byte fr die Timer-Frequenz }
  Port[$40] := Hi(Timer);                { High-Byte fr den Timer-Frequenz }
END;

PROCEDURE WriteDSP(Value : Byte);

VAR
  Count   : Word;                                              { ein Zhler }
  Writing : Boolean;       { TRUE, wenn der DSP zum Datenempfang bereit ist }

BEGIN
  Count := 0;
  Writing := FALSE;
  WHILE NOT Writing DO  { wiederholen, solange der DSP nicht empfangsbereit }
    BEGIN
      Inc(Count);                                          { Zhler erhhen }
      Writing :=                                      { DPS empfangsbereit? }
        (Port[SB_BaseAddress+$0C] AND $80) = $00;
      IF Count = 10000 THEN      { Hat der Zhler den Stand 10000 erreicht? }
        BEGIN             { ja, DSP kann nicht beschrieben werden -> Fehler }
          WriteLn;
          WriteLn('Fehler: Der DSP kann nicht beschrieben werden.');
          Halt;
        END;
    END;
  Port[SB_BaseAddress+$0C] := Value;                    { schreiben zum DSP }
END;

PROCEDURE WriteDirect(Value : Byte);

BEGIN
  WriteDSP(dsp_WriteDirectUnpack);                    { DSP-Kommando senden }
  WriteDSP(Value);                                    { Wert zum DSP senden }
END;

PROCEDURE Speaker_On;

BEGIN
  WriteDSP(dsp_SpeakerOn);
END;

PROCEDURE Speaker_Off;

BEGIN
  WriteDSP(dsp_SpeakerOff);
END;

PROCEDURE PlayDirect(DataPtr : Pointer; Size : Word; Freq : Word;
  Init : Boolean);

VAR
  TimerOld : Pointer;          { Zeiger auf die alte Timer-Interruptroutine }

BEGIN
  IF Size <> 0 THEN                                      { Blockgre <> 0? }
    BEGIN                                                              { ja }
      IF Init THEN
        InitDSP;                                       { DSP initialisieren }
      Speaker_On;                                        { Lautsprecher ein }
      GetIntVec($08, TimerOld);             { alten Timer-Interrupt sichern }
      SetTimerFreq(Freq);                       { Timer-Frequenz einstellen }
      BytePtr := DataPtr;
      Count_ := 0;
      Done := FALSE;
      BlockSize := Size;
      SetIntVec($08, @NewTimerInt);              { Timer-Interrupt umlenken }

      { wiederholen, bis der Datenblock abgespielt }
      { oder irgendeine Taste gedrckt wurde       }

      REPEAT
        IF KeyPressed THEN                                { Taste gedrckt? }
          Done := TRUE;                               { ja, Ausgabe beenden }
      UNTIL Done;

      SetOrigFreq;             { Original-Frequenz fr den Timer einstellen }
      SetIntVec($08, TimerOld);                 { alte Timer-Routine setzen }
      Speaker_Off;                                       { Lautsprecher aus }
    END;
END;

PROCEDURE Mixer(Stereo, Filter : Boolean);

VAR
  Help : Byte;                                              { Hilfsvariable }

BEGIN
  Port[SB_BaseAddress+$04] := $0E;                        { Kommando senden }
  Help := Port[SB_BaseAddress+$05] AND $DD;
  IF Stereo THEN                                                  { Stereo? }
    Help := Help OR $02;                                               { ja }
  IF NOT Filter THEN                                              { Filter? }
    Help := Help OR $20;                                               { ja }

  { Einstellungen setzen }

  Port[SB_BaseAddress+$04] := $0E;
  Port[SB_BaseAddress+$05] := Help;
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

BEGIN
  InitSBDrv;
END.
